/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.daemon.logviewer.utils;

import com.codahale.metrics.Meter;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Set;
import java.util.Stack;
import java.util.regex.Pattern;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.storm.daemon.logviewer.utils.DeletionMeta;
import org.apache.storm.metric.StormMetricsRegistry;
import org.apache.storm.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DirectoryCleaner {
    private static final Logger LOG = LoggerFactory.getLogger(DirectoryCleaner.class);
    private static final Pattern ACTIVE_LOG_PATTERN = Pattern.compile(".*\\.(log|err|out|current|yaml|pid|metrics)$");
    private static final Pattern META_LOG_PATTERN = Pattern.compile(".*\\.(yaml|pid)$");
    private static final int PQ_SIZE = 1024;
    private static final int MAX_ROUNDS = 512;
    public static final int MAX_NUMBER_OF_FILES_FOR_DIR = 1024;
    private final Meter numFileOpenExceptions;

    public DirectoryCleaner(StormMetricsRegistry metricsRegistry) {
        this.numFileOpenExceptions = metricsRegistry.registerMeter("logviewer:num-file-open-exceptions");
    }

    public DirectoryStream<Path> getStreamForDirectory(Path dir) throws IOException {
        try {
            return Files.newDirectoryStream(dir);
        }
        catch (IOException e) {
            this.numFileOpenExceptions.mark();
            throw e;
        }
    }

    public DeletionMeta deleteOldestWhileTooLarge(List<Path> dirs, long quota, boolean forPerDir, Set<Path> activeDirs) throws IOException {
        long totalSize = 0L;
        for (Path dir : dirs) {
            DirectoryStream<Path> stream = this.getStreamForDirectory(dir);
            Throwable throwable = null;
            try {
                for (Path path : stream) {
                    totalSize += Files.size(path);
                }
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (stream == null) continue;
                if (throwable != null) {
                    try {
                        stream.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                stream.close();
            }
        }
        LOG.debug("totalSize: {} quota: {}", (Object)totalSize, (Object)quota);
        long toDeleteSize = totalSize - quota;
        if (toDeleteSize <= 0L) {
            return DeletionMeta.EMPTY;
        }
        int deletedFiles = 0;
        long deletedSize = 0L;
        PriorityQueue<Pair> pq = new PriorityQueue<Pair>(1024, Comparator.comparing(p -> (FileTime)p.getRight()).reversed());
        int round = 0;
        HashSet<Path> excluded = new HashSet<Path>();
        while (toDeleteSize > 0L) {
            LOG.debug("To delete size is {}, start a new round of deletion, round: {}", (Object)toDeleteSize, (Object)round);
            for (Path dir : dirs) {
                DirectoryStream<Path> stream = this.getStreamForDirectory(dir);
                Throwable throwable = null;
                try {
                    for (Path path : stream) {
                        if (excluded.contains(path)) continue;
                        if (this.isFileEligibleToSkipDelete(forPerDir, activeDirs, dir, path)) {
                            excluded.add(path);
                            continue;
                        }
                        Pair p2 = Pair.of((Object)path, (Object)Files.getLastModifiedTime(path, new LinkOption[0]));
                        if (pq.size() < 1024) {
                            pq.offer(p2);
                            continue;
                        }
                        if (((FileTime)p2.getRight()).toMillis() >= ((FileTime)pq.peek().getRight()).toMillis()) continue;
                        pq.poll();
                        pq.offer(p2);
                    }
                }
                catch (Throwable throwable4) {
                    throwable = throwable4;
                    throw throwable4;
                }
                finally {
                    if (stream == null) continue;
                    if (throwable != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable5) {
                            throwable.addSuppressed(throwable5);
                        }
                        continue;
                    }
                    stream.close();
                }
            }
            if (!pq.isEmpty()) {
                Stack<Pair> stack = new Stack<Pair>();
                while (!pq.isEmpty()) {
                    stack.push(pq.poll());
                }
                while (!stack.isEmpty() && toDeleteSize > 0L) {
                    Pair pair = (Pair)stack.pop();
                    Path file = (Path)pair.getLeft();
                    String canonicalPath = file.toAbsolutePath().normalize().toString();
                    long fileSize = Files.size(file);
                    long lastModified = ((FileTime)pair.getRight()).toMillis();
                    try {
                        Utils.forceDelete((String)file.toString());
                        LOG.info("Delete file: {}, size: {}, lastModified: {}", new Object[]{canonicalPath, fileSize, lastModified});
                        toDeleteSize -= fileSize;
                        deletedSize += fileSize;
                        ++deletedFiles;
                    }
                    catch (IOException e) {
                        excluded.add(file);
                    }
                }
                pq.clear();
                if (++round < 512) continue;
                if (forPerDir) {
                    LOG.warn("Reach the MAX_ROUNDS: {} during per-dir deletion, you may have too many files in a single directory : {}, will delete the rest files in next interval.", (Object)512, (Object)dirs.get(0).toAbsolutePath().normalize());
                    break;
                }
                LOG.warn("Reach the MAX_ROUNDS: {} during global deletion, you may have too many files, will delete the rest files in next interval.", (Object)512);
                break;
            }
            LOG.warn("No more files able to delete this round, but {} is over quota by {} MB", (Object)(forPerDir ? "this directory" : "root directory"), (Object)((double)toDeleteSize * 1.0E-6));
            LOG.warn("No more files eligible to be deleted this round, but {} is over {} quota by {} MB", new Object[]{forPerDir ? "worker directory: " + dirs.get(0).toAbsolutePath().normalize() : "log root directory", forPerDir ? "per-worker" : "global", (double)toDeleteSize * 1.0E-6});
        }
        return new DeletionMeta(deletedSize, deletedFiles);
    }

    private boolean isFileEligibleToSkipDelete(boolean forPerDir, Set<Path> activeDirs, Path dir, Path file) throws IOException {
        if (forPerDir) {
            return ACTIVE_LOG_PATTERN.matcher(file.getFileName().toString()).matches();
        }
        return activeDirs.contains(dir) ? ACTIVE_LOG_PATTERN.matcher(file.getFileName().toString()).matches() : META_LOG_PATTERN.matcher(file.getFileName().toString()).matches();
    }

    public List<Path> getFilesForDir(Path dir) throws IOException {
        ArrayList<Path> files = new ArrayList<Path>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(dir);){
            for (Path path : stream) {
                files.add(path);
                if (files.size() < 1024) continue;
                break;
            }
        }
        catch (IOException e) {
            this.numFileOpenExceptions.mark();
            throw e;
        }
        return files;
    }
}

