/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.scheduler.resource.normalization;

import java.util.Arrays;
import java.util.Map;
import org.apache.storm.generated.WorkerResources;
import org.apache.storm.scheduler.resource.normalization.ResourceMapArrayBridge;
import org.apache.storm.scheduler.resource.normalization.ResourceMetrics;
import org.apache.storm.scheduler.resource.normalization.ResourceNameNormalizer;
import org.apache.storm.shade.com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NormalizedResources {
    private static final Logger LOG = LoggerFactory.getLogger(NormalizedResources.class);
    public static ResourceNameNormalizer RESOURCE_NAME_NORMALIZER;
    private static ResourceMapArrayBridge RESOURCE_MAP_ARRAY_BRIDGE;
    private double cpu;
    private double[] otherResources;

    NormalizedResources() {
        this.cpu = 0.0;
        this.otherResources = RESOURCE_MAP_ARRAY_BRIDGE.empty();
    }

    public NormalizedResources(NormalizedResources other) {
        this.cpu = other.cpu;
        this.otherResources = Arrays.copyOf(other.otherResources, other.otherResources.length);
    }

    public NormalizedResources(Map<String, Double> normalizedResources) {
        this.cpu = normalizedResources.getOrDefault("cpu.pcore.percent", 0.0);
        this.otherResources = RESOURCE_MAP_ARRAY_BRIDGE.translateToResourceArray(normalizedResources);
    }

    @VisibleForTesting
    public static void resetResourceNames() {
        RESOURCE_NAME_NORMALIZER = new ResourceNameNormalizer();
        RESOURCE_MAP_ARRAY_BRIDGE = new ResourceMapArrayBridge();
    }

    public double getTotalCpu() {
        return this.cpu;
    }

    private void zeroPadOtherResourcesIfNecessary(int requiredLength) {
        if (requiredLength > this.otherResources.length) {
            double[] newResources = new double[requiredLength];
            System.arraycopy(this.otherResources, 0, newResources, 0, this.otherResources.length);
            this.otherResources = newResources;
        }
    }

    private void add(double[] resourceArray) {
        int otherLength = resourceArray.length;
        this.zeroPadOtherResourcesIfNecessary(otherLength);
        for (int i = 0; i < otherLength; ++i) {
            int n = i;
            this.otherResources[n] = this.otherResources[n] + resourceArray[i];
        }
    }

    public void add(NormalizedResources other) {
        this.cpu += other.cpu;
        this.add(other.otherResources);
    }

    public void add(WorkerResources value) {
        Map workerNormalizedResources = value.get_resources();
        this.cpu += workerNormalizedResources.getOrDefault("cpu.pcore.percent", 0.0).doubleValue();
        this.add(RESOURCE_MAP_ARRAY_BRIDGE.translateToResourceArray(workerNormalizedResources));
    }

    public boolean remove(NormalizedResources other, ResourceMetrics resourceMetrics) {
        boolean ret = false;
        this.cpu -= other.cpu;
        if (this.cpu < 0.0) {
            ret = true;
            if (resourceMetrics != null) {
                resourceMetrics.getNegativeResourceEventsMeter().mark();
            }
            this.cpu = 0.0;
        }
        int otherLength = other.otherResources.length;
        this.zeroPadOtherResourcesIfNecessary(otherLength);
        for (int i = 0; i < otherLength; ++i) {
            int n = i;
            this.otherResources[n] = this.otherResources[n] - other.otherResources[i];
            if (!(this.otherResources[i] < 0.0)) continue;
            ret = true;
            if (resourceMetrics != null) {
                resourceMetrics.getNegativeResourceEventsMeter().mark();
            }
            this.otherResources[i] = 0.0;
        }
        return ret;
    }

    public boolean remove(WorkerResources value) {
        Map workerNormalizedResources = value.get_resources();
        this.cpu -= workerNormalizedResources.getOrDefault("cpu.pcore.percent", 0.0).doubleValue();
        return this.remove(RESOURCE_MAP_ARRAY_BRIDGE.translateToResourceArray(workerNormalizedResources)) || this.cpu < 0.0;
    }

    private boolean remove(double[] resourceArray) {
        boolean ret = false;
        int otherLength = resourceArray.length;
        this.zeroPadOtherResourcesIfNecessary(otherLength);
        for (int i = 0; i < otherLength; ++i) {
            int n = i;
            this.otherResources[n] = this.otherResources[n] - resourceArray[i];
            if (!(this.otherResources[i] < 0.0)) continue;
            ret = true;
        }
        return ret;
    }

    public String toString() {
        return "Normalized resources: " + this.toNormalizedMap();
    }

    public Map<String, Double> toNormalizedMap() {
        Map<String, Double> ret = RESOURCE_MAP_ARRAY_BRIDGE.translateFromResourceArray(this.otherResources);
        ret.put("cpu.pcore.percent", this.cpu);
        return ret;
    }

    private double getResourceAt(int index) {
        if (index >= this.otherResources.length) {
            return 0.0;
        }
        return this.otherResources[index];
    }

    public boolean couldHoldIgnoringSharedMemory(NormalizedResources other, double thisTotalMemoryMb, double otherTotalMemoryMb) {
        if (this.cpu < other.getTotalCpu()) {
            return false;
        }
        return this.couldHoldIgnoringSharedMemoryAndCpu(other, thisTotalMemoryMb, otherTotalMemoryMb);
    }

    public boolean couldHoldIgnoringSharedMemoryAndCpu(NormalizedResources other, double thisTotalMemoryMb, double otherTotalMemoryMb) {
        int length = Math.max(this.otherResources.length, other.otherResources.length);
        for (int i = 0; i < length; ++i) {
            if (!(this.getResourceAt(i) < other.getResourceAt(i))) continue;
            return false;
        }
        return thisTotalMemoryMb >= otherTotalMemoryMb;
    }

    private String getResourceNameForResourceIndex(int resourceIndex) {
        for (Map.Entry<String, Integer> entry : RESOURCE_MAP_ARRAY_BRIDGE.getResourceNamesToArrayIndex().entrySet()) {
            int index = entry.getValue();
            if (index != resourceIndex) continue;
            return entry.getKey();
        }
        return null;
    }

    private void throwBecauseUsedIsNotSubsetOfTotal(NormalizedResources used, double totalMemoryMb, double usedMemoryMb) {
        throw new IllegalArgumentException(String.format("The used resources must be a subset of the total resources. Used: '%s', Total: '%s', Used Mem: '%f', Total Mem: '%f'", used.toNormalizedMap(), this.toNormalizedMap(), usedMemoryMb, totalMemoryMb));
    }

    public double calculateAveragePercentageUsedBy(NormalizedResources used, double totalMemoryMb, double usedMemoryMb) {
        int skippedResourceTypes = 0;
        double total = 0.0;
        if (usedMemoryMb > totalMemoryMb) {
            this.throwBecauseUsedIsNotSubsetOfTotal(used, totalMemoryMb, usedMemoryMb);
        }
        if (totalMemoryMb != 0.0) {
            total += usedMemoryMb / totalMemoryMb;
        } else {
            ++skippedResourceTypes;
        }
        double totalCpu = this.getTotalCpu();
        if (used.getTotalCpu() > this.getTotalCpu()) {
            this.throwBecauseUsedIsNotSubsetOfTotal(used, totalMemoryMb, usedMemoryMb);
        }
        if (totalCpu != 0.0) {
            total += used.getTotalCpu() / this.getTotalCpu();
        } else {
            ++skippedResourceTypes;
        }
        if (used.otherResources.length > this.otherResources.length) {
            this.throwBecauseUsedIsNotSubsetOfTotal(used, totalMemoryMb, usedMemoryMb);
        }
        for (int i = 0; i < this.otherResources.length; ++i) {
            double totalValue;
            double usedValue = i >= used.otherResources.length ? 0.0 : used.otherResources[i];
            if (usedValue > (totalValue = this.otherResources[i])) {
                this.throwBecauseUsedIsNotSubsetOfTotal(used, totalMemoryMb, usedMemoryMb);
            }
            if (totalValue == 0.0) {
                ++skippedResourceTypes;
                continue;
            }
            total += usedValue / totalValue;
        }
        int divisor = 2 + this.otherResources.length - skippedResourceTypes;
        if (divisor == 0) {
            return 100.0;
        }
        return total * 100.0 / (double)divisor;
    }

    public double calculateMinPercentageUsedBy(NormalizedResources used, double totalMemoryMb, double usedMemoryMb) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("Calculating min percentage used by. Used Mem: {} Total Mem: {} Used Normalized Resources: {} Total Normalized Resources: {}", new Object[]{totalMemoryMb, usedMemoryMb, this.toNormalizedMap(), used.toNormalizedMap()});
        }
        double min = 1.0;
        if (usedMemoryMb > totalMemoryMb) {
            this.throwBecauseUsedIsNotSubsetOfTotal(used, totalMemoryMb, usedMemoryMb);
        }
        if (totalMemoryMb != 0.0) {
            min = Math.min(min, usedMemoryMb / totalMemoryMb);
        }
        double totalCpu = this.getTotalCpu();
        if (used.getTotalCpu() > totalCpu) {
            this.throwBecauseUsedIsNotSubsetOfTotal(used, totalMemoryMb, usedMemoryMb);
        }
        if (totalCpu != 0.0) {
            min = Math.min(min, used.getTotalCpu() / totalCpu);
        }
        if (used.otherResources.length > this.otherResources.length) {
            this.throwBecauseUsedIsNotSubsetOfTotal(used, totalMemoryMb, usedMemoryMb);
        }
        for (int i = 0; i < this.otherResources.length; ++i) {
            if (this.otherResources[i] == 0.0) continue;
            if (i >= used.otherResources.length) {
                return 0.0;
            }
            if (used.otherResources[i] > this.otherResources[i]) {
                this.throwBecauseUsedIsNotSubsetOfTotal(used, totalMemoryMb, usedMemoryMb);
            }
            min = Math.min(min, used.otherResources[i] / this.otherResources[i]);
        }
        return min * 100.0;
    }

    public void updateForRareResourceAffinity(NormalizedResources request) {
        int length = Math.min(this.otherResources.length, request.otherResources.length);
        for (int i = 0; i < length; ++i) {
            if (request.getResourceAt(i) != 0.0) continue;
            this.otherResources[i] = -1.0 * this.otherResources[i];
        }
    }

    public void clear() {
        this.cpu = 0.0;
        for (int i = 0; i < this.otherResources.length; ++i) {
            this.otherResources[i] = 0.0;
        }
    }

    private boolean areAnyOverZero(boolean skipCpuCheck) {
        for (int i = 0; i < this.otherResources.length; ++i) {
            if (!(this.otherResources[i] > 0.0)) continue;
            return true;
        }
        if (skipCpuCheck) {
            return false;
        }
        return this.cpu > 0.0;
    }

    public boolean areAnyOverZero() {
        return this.areAnyOverZero(false);
    }

    public boolean anyNonCpuOverZero() {
        return this.areAnyOverZero(true);
    }

    static {
        NormalizedResources.resetResourceNames();
    }
}

