/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.container.cgroup;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.SystemUtils;
import org.apache.storm.DaemonConfig;
import org.apache.storm.container.ResourceIsolationInterface;
import org.apache.storm.container.cgroup.CgroupCenter;
import org.apache.storm.container.cgroup.CgroupCommon;
import org.apache.storm.container.cgroup.Hierarchy;
import org.apache.storm.container.cgroup.SubSystemType;
import org.apache.storm.container.cgroup.core.CpuCore;
import org.apache.storm.container.cgroup.core.CpusetCore;
import org.apache.storm.container.cgroup.core.MemoryCore;
import org.apache.storm.utils.ObjectReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CgroupManager
implements ResourceIsolationInterface {
    private static final Logger LOG = LoggerFactory.getLogger(CgroupManager.class);
    private static final Pattern MEMINFO_PATTERN = Pattern.compile("^([^:\\s]+):\\s*([0-9]+)\\s*kB$");
    private CgroupCenter center;
    private Hierarchy hierarchy;
    private CgroupCommon rootCgroup;
    private String rootDir;
    private Map<String, Object> conf;
    private Map<String, String> workerToNumaId;

    static long getMemInfoFreeMb() throws IOException {
        long memFree = 0L;
        long buffers = 0L;
        long cached = 0L;
        try (BufferedReader in = new BufferedReader(new FileReader("/proc/meminfo"));){
            String line = null;
            while ((line = in.readLine()) != null) {
                Matcher match = MEMINFO_PATTERN.matcher(line);
                if (!match.matches()) continue;
                String tag = match.group(1);
                if (tag.equalsIgnoreCase("MemFree")) {
                    memFree = Long.parseLong(match.group(2));
                    continue;
                }
                if (tag.equalsIgnoreCase("Buffers")) {
                    buffers = Long.parseLong(match.group(2));
                    continue;
                }
                if (!tag.equalsIgnoreCase("Cached")) continue;
                cached = Long.parseLong(match.group(2));
            }
        }
        return (memFree + buffers + cached) / 1024L;
    }

    @Override
    public void prepare(Map<String, Object> conf) throws IOException {
        this.conf = conf;
        this.rootDir = DaemonConfig.getCgroupRootDir(this.conf);
        if (this.rootDir == null) {
            throw new RuntimeException("Check configuration file. The storm.supervisor.cgroup.rootdir is missing.");
        }
        File file = new File(DaemonConfig.getCgroupStormHierarchyDir(conf), this.rootDir);
        if (!file.exists()) {
            LOG.error("{} does not exist", (Object)file.getPath());
            throw new RuntimeException("Check if cgconfig service starts or /etc/cgconfig.conf is consistent with configuration file.");
        }
        this.center = CgroupCenter.getInstance();
        if (this.center == null) {
            throw new RuntimeException("Cgroup error, please check /proc/cgroups");
        }
        this.prepareSubSystem(this.conf);
        this.workerToNumaId = new ConcurrentHashMap<String, String>();
    }

    private void prepareSubSystem(Map<String, Object> conf) throws IOException {
        LinkedList<SubSystemType> subSystemTypes = new LinkedList<SubSystemType>();
        for (String resource : DaemonConfig.getCgroupStormResources(conf)) {
            subSystemTypes.add(SubSystemType.getSubSystem((String)resource));
        }
        this.hierarchy = this.center.getHierarchyWithSubSystems(subSystemTypes);
        if (this.hierarchy == null) {
            HashSet<SubSystemType> types = new HashSet<SubSystemType>();
            types.add(SubSystemType.cpu);
            this.hierarchy = new Hierarchy(DaemonConfig.getCgroupStormHierarchyName(conf), types, DaemonConfig.getCgroupStormHierarchyDir(conf));
        }
        this.rootCgroup = new CgroupCommon(this.rootDir, this.hierarchy, this.hierarchy.getRootCgroups());
        CpuCore supervisorRootCpu = (CpuCore)this.rootCgroup.getCores().get(SubSystemType.cpu);
        this.setCpuUsageUpperLimit(supervisorRootCpu, ((Number)this.conf.get("supervisor.cpu.capacity")).intValue());
    }

    private void setCpuUsageUpperLimit(CpuCore cpuCore, int cpuCoreUpperLimit) throws IOException {
        if (cpuCoreUpperLimit == -1) {
            cpuCore.setCpuCfsQuotaUs((long)cpuCoreUpperLimit);
        } else {
            cpuCore.setCpuCfsPeriodUs(100000L);
            cpuCore.setCpuCfsQuotaUs((long)(cpuCoreUpperLimit * 1000));
        }
    }

    @Override
    public void reserveResourcesForWorker(String workerId, Integer totalMem, Integer cpuNum, String numaId) throws SecurityException {
        LOG.info("Creating cgroup for worker {} with resources {} MB {} % CPU", new Object[]{workerId, totalMem, cpuNum});
        if (this.conf.get(DaemonConfig.STORM_WORKER_CGROUP_CPU_LIMIT) != null) {
            cpuNum = ((Number)this.conf.get(DaemonConfig.STORM_WORKER_CGROUP_CPU_LIMIT)).intValue();
        }
        if (this.conf.get(DaemonConfig.STORM_WORKER_CGROUP_MEMORY_MB_LIMIT) != null) {
            totalMem = ((Number)this.conf.get(DaemonConfig.STORM_WORKER_CGROUP_MEMORY_MB_LIMIT)).intValue();
        }
        CgroupCommon workerGroup = new CgroupCommon(workerId, this.hierarchy, this.rootCgroup);
        try {
            this.center.createCgroup(workerGroup);
        }
        catch (Exception e) {
            throw new RuntimeException("Error when creating Cgroup! Exception: ", e);
        }
        if (cpuNum != null) {
            CpuCore cpuCore = (CpuCore)workerGroup.getCores().get(SubSystemType.cpu);
            try {
                cpuCore.setCpuShares(cpuNum.intValue());
            }
            catch (IOException e) {
                throw new RuntimeException("Cannot set cpu.shares! Exception: ", e);
            }
        }
        if (((Boolean)this.conf.get(DaemonConfig.STORM_CGROUP_MEMORY_ENFORCEMENT_ENABLE)).booleanValue() && totalMem != null) {
            int cgroupMem = (int)Math.ceil(ObjectReader.getDouble((Object)this.conf.get(DaemonConfig.STORM_CGROUP_MEMORY_LIMIT_TOLERANCE_MARGIN_MB), (Double)0.0));
            long memLimit = (totalMem.longValue() + (long)cgroupMem) * 1024L * 1024L;
            MemoryCore memCore = (MemoryCore)workerGroup.getCores().get(SubSystemType.memory);
            try {
                memCore.setPhysicalUsageLimit(memLimit);
            }
            catch (IOException e) {
                throw new RuntimeException("Cannot set memory.limit_in_bytes! Exception: ", e);
            }
            try {
                memCore.setWithSwapUsageLimit(memLimit);
            }
            catch (IOException e) {
                throw new RuntimeException("Cannot set memory.memsw.limit_in_bytes! Exception: ", e);
            }
        }
        if (((Boolean)this.conf.get(DaemonConfig.STORM_CGROUP_INHERIT_CPUSET_CONFIGS)).booleanValue() && workerGroup.getParent().getCores().containsKey(SubSystemType.cpuset)) {
            CpusetCore parentCpusetCore = (CpusetCore)workerGroup.getParent().getCores().get(SubSystemType.cpuset);
            CpusetCore cpusetCore = (CpusetCore)workerGroup.getCores().get(SubSystemType.cpuset);
            try {
                cpusetCore.setCpus(parentCpusetCore.getCpus());
            }
            catch (IOException e) {
                throw new RuntimeException("Cannot set cpuset.cpus! Exception: ", e);
            }
            try {
                cpusetCore.setMems(parentCpusetCore.getMems());
            }
            catch (IOException e) {
                throw new RuntimeException("Cannot set cpuset.mems! Exception: ", e);
            }
        }
        if (numaId != null) {
            this.workerToNumaId.put(workerId, numaId);
        }
    }

    @Override
    public void releaseResourcesForWorker(String workerId) {
        CgroupCommon workerGroup = new CgroupCommon(workerId, this.hierarchy, this.rootCgroup);
        try {
            Set tasks = workerGroup.getTasks();
            if (!tasks.isEmpty()) {
                throw new Exception("Cannot correctly shutdown worker CGroup " + workerId + "tasks " + tasks + " still running!");
            }
            this.center.deleteCgroup(workerGroup);
        }
        catch (Exception e) {
            LOG.error("Exception thrown when shutting worker {} Exception: {}", (Object)workerId, (Object)e);
        }
    }

    protected static boolean isOnLinux() {
        return SystemUtils.IS_OS_LINUX;
    }

    private void prefixNumaPinning(List<String> command, String numaId) {
        if (CgroupManager.isOnLinux()) {
            command.add(0, "numactl");
            command.add(1, "--cpunodebind=" + numaId);
            command.add(2, "--membind=" + numaId);
            return;
        }
        throw new RuntimeException("numactl pinning currently not supported on non-Linux hosts");
    }

    @Override
    public List<String> getLaunchCommand(String workerId, List<String> existingCommand) {
        List<String> newCommand = this.getLaunchCommandPrefix(workerId);
        if (this.workerToNumaId.containsKey(workerId)) {
            this.prefixNumaPinning(newCommand, this.workerToNumaId.get(workerId));
        }
        newCommand.addAll(existingCommand);
        return newCommand;
    }

    @Override
    public List<String> getLaunchCommandPrefix(String workerId) {
        CgroupCommon workerGroup = new CgroupCommon(workerId, this.hierarchy, this.rootCgroup);
        if (!this.rootCgroup.getChildren().contains(workerGroup)) {
            throw new RuntimeException("cgroup " + workerGroup + " doesn't exist! Need to reserve resources for worker first!");
        }
        StringBuilder sb = new StringBuilder();
        sb.append(this.conf.get(DaemonConfig.STORM_CGROUP_CGEXEC_CMD)).append(" -g ");
        Iterator it = this.hierarchy.getSubSystems().iterator();
        while (it.hasNext()) {
            sb.append(((SubSystemType)it.next()).toString());
            if (it.hasNext()) {
                sb.append(",");
                continue;
            }
            sb.append(":");
        }
        sb.append(workerGroup.getName());
        ArrayList<String> newCommand = new ArrayList<String>();
        newCommand.addAll(Arrays.asList(sb.toString().split(" ")));
        return newCommand;
    }

    @Override
    public Set<Long> getRunningPids(String workerId) throws IOException {
        CgroupCommon workerGroup = new CgroupCommon(workerId, this.hierarchy, this.rootCgroup);
        if (!this.rootCgroup.getChildren().contains(workerGroup)) {
            LOG.warn("cgroup {} doesn't exist!", (Object)workerGroup);
            return Collections.emptySet();
        }
        return workerGroup.getPids();
    }

    @Override
    public long getMemoryUsage(String workerId) throws IOException {
        CgroupCommon workerGroup = new CgroupCommon(workerId, this.hierarchy, this.rootCgroup);
        MemoryCore memCore = (MemoryCore)workerGroup.getCores().get(SubSystemType.memory);
        return memCore.getPhysicalUsage();
    }

    @Override
    public long getSystemFreeMemoryMb() throws IOException {
        long rootCgroupLimitFree = Long.MAX_VALUE;
        try {
            MemoryCore memRoot = (MemoryCore)this.rootCgroup.getCores().get(SubSystemType.memory);
            if (memRoot != null) {
                long limit = memRoot.getPhysicalUsageLimit();
                long used = memRoot.getMaxPhysicalUsage();
                rootCgroupLimitFree = (limit - used) / 1024L / 1024L;
            }
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
        return Long.min(rootCgroupLimitFree, CgroupManager.getMemInfoFreeMb());
    }
}

