/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.blobstore;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import javax.security.auth.Subject;
import org.apache.storm.blobstore.AtomicOutputStream;
import org.apache.storm.blobstore.BlobStore;
import org.apache.storm.blobstore.BlobStoreAclHandler;
import org.apache.storm.blobstore.BlobStoreFile;
import org.apache.storm.blobstore.BlobStoreUtils;
import org.apache.storm.blobstore.FileBlobStoreImpl;
import org.apache.storm.blobstore.InputStreamWithMeta;
import org.apache.storm.blobstore.LocalFsBlobStoreFile;
import org.apache.storm.blobstore.LocalFsBlobStoreSynchronizer;
import org.apache.storm.cluster.ClusterStateContext;
import org.apache.storm.cluster.ClusterUtils;
import org.apache.storm.cluster.DaemonType;
import org.apache.storm.cluster.IStormClusterState;
import org.apache.storm.daemon.nimbus.Nimbus;
import org.apache.storm.generated.AuthorizationException;
import org.apache.storm.generated.KeyAlreadyExistsException;
import org.apache.storm.generated.KeyNotFoundException;
import org.apache.storm.generated.ReadableBlobMeta;
import org.apache.storm.generated.SettableBlobMeta;
import org.apache.storm.nimbus.ILeaderElector;
import org.apache.storm.nimbus.NimbusInfo;
import org.apache.storm.shade.com.google.common.annotations.VisibleForTesting;
import org.apache.storm.shade.org.apache.curator.framework.CuratorFramework;
import org.apache.storm.shade.org.apache.zookeeper.KeeperException;
import org.apache.storm.thrift.TBase;
import org.apache.storm.utils.ConfigUtils;
import org.apache.storm.utils.ObjectReader;
import org.apache.storm.utils.Utils;
import org.apache.storm.utils.WrappedKeyAlreadyExistsException;
import org.apache.storm.utils.WrappedKeyNotFoundException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocalFsBlobStore
extends BlobStore {
    public static final Logger LOG = LoggerFactory.getLogger(LocalFsBlobStore.class);
    private static final String DATA_PREFIX = "data_";
    private static final String META_PREFIX = "meta_";
    private static final String BLOBSTORE_SUBTREE = "/blobstore/";
    private final int allPermissions = 7;
    protected BlobStoreAclHandler aclHandler;
    private NimbusInfo nimbusInfo;
    private FileBlobStoreImpl fbs;
    private Map<String, Object> conf;
    private CuratorFramework zkClient;
    private IStormClusterState stormClusterState;
    private Timer timer;
    private ILeaderElector leaderElector;

    public void prepare(Map<String, Object> conf, String overrideBase, NimbusInfo nimbusInfo, ILeaderElector leaderElector) {
        this.conf = conf;
        this.nimbusInfo = nimbusInfo;
        this.zkClient = BlobStoreUtils.createZKClient(conf, DaemonType.NIMBUS);
        if (overrideBase == null) {
            overrideBase = ConfigUtils.absoluteStormBlobStoreDir(conf);
        }
        File baseDir = new File(overrideBase, "blobs");
        try {
            this.fbs = new FileBlobStoreImpl(baseDir, conf);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        this.aclHandler = new BlobStoreAclHandler(conf);
        try {
            this.stormClusterState = ClusterUtils.mkStormClusterState(conf, (ClusterStateContext)new ClusterStateContext(DaemonType.NIMBUS, conf));
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        this.timer = new Timer("BLOB-STORE-TIMER", true);
        this.leaderElector = leaderElector;
    }

    private void setupBlobstore() throws AuthorizationException, KeyNotFoundException {
        IStormClusterState state = this.stormClusterState;
        LocalFsBlobStore store = this;
        HashSet localKeys = new HashSet();
        Iterator it = store.listKeys();
        while (it.hasNext()) {
            localKeys.add(it.next());
        }
        HashSet activeKeys = new HashSet(state.activeKeys());
        HashSet activeLocalKeys = new HashSet(localKeys);
        activeLocalKeys.retainAll(activeKeys);
        HashSet keysToDelete = new HashSet(localKeys);
        keysToDelete.removeAll(activeKeys);
        NimbusInfo nimbusInfo = this.nimbusInfo;
        LOG.debug("Deleting keys not on the zookeeper {}", keysToDelete);
        for (String toDelete : keysToDelete) {
            store.deleteBlob(toDelete, Nimbus.NIMBUS_SUBJECT);
        }
        LOG.debug("Creating list of key entries for blobstore inside zookeeper {} local {}", activeKeys, activeLocalKeys);
        for (String key : activeLocalKeys) {
            try {
                state.setupBlob(key, nimbusInfo, Integer.valueOf(Nimbus.getVersionForKey(key, nimbusInfo, this.zkClient)));
            }
            catch (KeyNotFoundException e) {
                store.deleteBlob(key, Nimbus.NIMBUS_SUBJECT);
            }
        }
    }

    private void blobSync() throws Exception {
        if ("distributed".equals(this.conf.get("storm.cluster.mode")) && !this.leaderElector.isLeader()) {
            IStormClusterState state = this.stormClusterState;
            NimbusInfo nimbusInfo = this.nimbusInfo;
            LocalFsBlobStore store = this;
            HashSet<String> allKeys = new HashSet<String>();
            Iterator it = store.listKeys();
            while (it.hasNext()) {
                allKeys.add((String)it.next());
            }
            HashSet<String> zkKeys = new HashSet<String>(state.blobstore(() -> {
                try {
                    this.blobSync();
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }));
            LOG.debug("blob-sync blob-store-keys {} zookeeper-keys {}", allKeys, zkKeys);
            LocalFsBlobStoreSynchronizer sync = new LocalFsBlobStoreSynchronizer(store, this.conf);
            sync.setNimbusInfo(nimbusInfo);
            sync.setBlobStoreKeySet(allKeys);
            sync.setZookeeperKeySet(zkKeys);
            sync.setZkClient(this.zkClient);
            sync.syncBlobs();
        }
    }

    public void startSyncBlobs() throws KeyNotFoundException, AuthorizationException {
        this.stormClusterState.blobstore(() -> {
            try {
                this.blobSync();
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
        this.setupBlobstore();
        this.timer.scheduleAtFixedRate(new TimerTask(){

            @Override
            public void run() {
                try {
                    LocalFsBlobStore.this.blobSync();
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
        }, 0L, (long)(ObjectReader.getInt((Object)this.conf.get("nimbus.code.sync.freq.secs")) * 1000));
    }

    public AtomicOutputStream createBlob(String key, SettableBlobMeta meta, Subject who) throws AuthorizationException, KeyAlreadyExistsException {
        LOG.debug("Creating Blob for key {}", (Object)key);
        LocalFsBlobStore.validateKey((String)key);
        this.aclHandler.normalizeSettableBlobMeta(key, meta, who, 7);
        BlobStoreAclHandler.validateSettableACLs((String)key, (List)meta.get_acl());
        this.aclHandler.hasPermissions(meta.get_acl(), 7, who, key);
        if (this.fbs.exists(DATA_PREFIX + key)) {
            throw new WrappedKeyAlreadyExistsException(key);
        }
        BlobStore.BlobStoreFileOutputStream outputStream = null;
        try {
            outputStream = new BlobStore.BlobStoreFileOutputStream((BlobStore)this, (BlobStoreFile)this.fbs.write(META_PREFIX + key, true));
            outputStream.write(Utils.thriftSerialize((TBase)meta));
            outputStream.close();
            outputStream = null;
            this.stormClusterState.setupBlob(key, this.nimbusInfo, Integer.valueOf(Nimbus.getVersionForKey(key, this.nimbusInfo, this.zkClient)));
            BlobStore.BlobStoreFileOutputStream blobStoreFileOutputStream = new BlobStore.BlobStoreFileOutputStream((BlobStore)this, (BlobStoreFile)this.fbs.write(DATA_PREFIX + key, true));
            return blobStoreFileOutputStream;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        catch (KeyNotFoundException e) {
            throw new RuntimeException(e);
        }
        finally {
            if (outputStream != null) {
                try {
                    outputStream.cancel();
                }
                catch (IOException iOException) {}
            }
        }
    }

    public AtomicOutputStream updateBlob(String key, Subject who) throws AuthorizationException, KeyNotFoundException {
        LocalFsBlobStore.validateKey((String)key);
        this.checkPermission(key, who, 2);
        try {
            return new BlobStore.BlobStoreFileOutputStream((BlobStore)this, (BlobStoreFile)this.fbs.write(DATA_PREFIX + key, false));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private SettableBlobMeta getStoredBlobMeta(String key) throws KeyNotFoundException {
        InputStream in = null;
        try {
            int len;
            LocalFsBlobStoreFile pf = this.fbs.read(META_PREFIX + key);
            try {
                in = pf.getInputStream();
            }
            catch (FileNotFoundException fnf) {
                throw new WrappedKeyNotFoundException(key);
            }
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            byte[] buffer = new byte[2048];
            while ((len = in.read(buffer)) > 0) {
                out.write(buffer, 0, len);
            }
            in.close();
            in = null;
            SettableBlobMeta settableBlobMeta = (SettableBlobMeta)Utils.thriftDeserialize(SettableBlobMeta.class, (byte[])out.toByteArray());
            return settableBlobMeta;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        finally {
            if (in != null) {
                try {
                    in.close();
                }
                catch (IOException iOException) {}
            }
        }
    }

    public ReadableBlobMeta getBlobMeta(String key, Subject who) throws AuthorizationException, KeyNotFoundException {
        LocalFsBlobStore.validateKey((String)key);
        if (!this.checkForBlobOrDownload(key)) {
            this.checkForBlobUpdate(key);
        }
        SettableBlobMeta meta = this.getStoredBlobMeta(key);
        this.aclHandler.validateUserCanReadMeta(meta.get_acl(), who, key);
        ReadableBlobMeta rbm = new ReadableBlobMeta();
        rbm.set_settable(meta);
        try {
            LocalFsBlobStoreFile pf = this.fbs.read(DATA_PREFIX + key);
            rbm.set_version(pf.getModTime());
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        return rbm;
    }

    public void setLeaderElector(ILeaderElector leaderElector) {
        this.leaderElector = leaderElector;
    }

    public void setBlobMeta(String key, SettableBlobMeta meta, Subject who) throws AuthorizationException, KeyNotFoundException {
        LocalFsBlobStore.validateKey((String)key);
        this.checkForBlobOrDownload(key);
        this.aclHandler.normalizeSettableBlobMeta(key, meta, who, 4);
        BlobStoreAclHandler.validateSettableACLs((String)key, (List)meta.get_acl());
        SettableBlobMeta orig = this.getStoredBlobMeta(key);
        this.aclHandler.hasPermissions(orig.get_acl(), 4, who, key);
        BlobStore.BlobStoreFileOutputStream outputStream = null;
        try {
            outputStream = new BlobStore.BlobStoreFileOutputStream((BlobStore)this, (BlobStoreFile)this.fbs.write(META_PREFIX + key, false));
            outputStream.write(Utils.thriftSerialize((TBase)meta));
            outputStream.close();
            outputStream = null;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        finally {
            if (outputStream != null) {
                try {
                    outputStream.cancel();
                }
                catch (IOException iOException) {}
            }
        }
    }

    public void deleteBlob(String key, Subject who) throws AuthorizationException, KeyNotFoundException {
        LocalFsBlobStore.validateKey((String)key);
        if (!this.aclHandler.checkForValidUsers(who, 2)) {
            LOG.debug("Retrieving meta to get ACL info... key: {} subject: {}", (Object)key, (Object)who);
            try {
                this.checkPermission(key, who, 2);
            }
            catch (KeyNotFoundException e) {
                LOG.error("Error while retrieving meta from ZK or local... key: {} subject: {}", (Object)key, (Object)who);
                throw e;
            }
        } else {
            LOG.debug("Given subject is eligible to delete key without checking ACL, skipping... key: {} subject: {}", (Object)key, (Object)who);
        }
        try {
            this.deleteKeyIgnoringFileNotFound(DATA_PREFIX + key);
            this.deleteKeyIgnoringFileNotFound(META_PREFIX + key);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        this.stormClusterState.removeBlobstoreKey(key);
        this.stormClusterState.removeKeyVersion(key);
    }

    private void checkPermission(String key, Subject who, int mask) throws KeyNotFoundException, AuthorizationException {
        this.checkForBlobOrDownload(key);
        SettableBlobMeta meta = this.getStoredBlobMeta(key);
        this.aclHandler.hasPermissions(meta.get_acl(), mask, who, key);
    }

    private void deleteKeyIgnoringFileNotFound(String key) throws IOException {
        try {
            this.fbs.deleteKey(key);
        }
        catch (IOException e) {
            if (e instanceof FileNotFoundException || e instanceof NoSuchFileException) {
                LOG.debug("Ignoring FileNotFoundException since we're about to delete such key... key: {}", (Object)key);
            }
            throw e;
        }
    }

    public InputStreamWithMeta getBlob(String key, Subject who) throws AuthorizationException, KeyNotFoundException {
        LocalFsBlobStore.validateKey((String)key);
        if (!this.checkForBlobOrDownload(key)) {
            this.checkForBlobUpdate(key);
        }
        SettableBlobMeta meta = this.getStoredBlobMeta(key);
        this.aclHandler.hasPermissions(meta.get_acl(), 1, who, key);
        try {
            return new BlobStore.BlobStoreFileInputStream((BlobStore)this, (BlobStoreFile)this.fbs.read(DATA_PREFIX + key));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public Iterator<String> listKeys() {
        try {
            return new BlobStore.KeyTranslationIterator(this.fbs.listKeys(), DATA_PREFIX);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public void shutdown() {
        if (this.zkClient != null) {
            this.zkClient.close();
        }
        if (this.timer != null) {
            this.timer.cancel();
        }
        this.stormClusterState.disconnect();
    }

    public int getBlobReplication(String key, Subject who) throws Exception {
        int replicationCount = 0;
        LocalFsBlobStore.validateKey((String)key);
        SettableBlobMeta meta = this.getStoredBlobMeta(key);
        this.aclHandler.hasPermissions(meta.get_acl(), 1, who, key);
        if (this.zkClient.checkExists().forPath(BLOBSTORE_SUBTREE + key) == null) {
            return 0;
        }
        try {
            replicationCount = ((List)this.zkClient.getChildren().forPath(BLOBSTORE_SUBTREE + key)).size();
        }
        catch (KeeperException.NoNodeException noNodeException) {
            // empty catch block
        }
        return replicationCount;
    }

    public int updateBlobReplication(String key, int replication, Subject who) throws AuthorizationException, KeyNotFoundException {
        throw new UnsupportedOperationException("For local file system blob store the update blobs function does not work. Please use HDFS blob store to make this feature available.");
    }

    public synchronized boolean checkForBlobOrDownload(String key) throws KeyNotFoundException {
        boolean checkBlobDownload = false;
        try {
            List<String> keyList = BlobStoreUtils.getKeyListFromBlobStore(this);
            if (!keyList.contains(key) && this.zkClient.checkExists().forPath(BLOBSTORE_SUBTREE + key) != null) {
                Set<NimbusInfo> nimbusSet = BlobStoreUtils.getNimbodesWithLatestSequenceNumberOfBlob(this.zkClient, key);
                nimbusSet.remove(this.nimbusInfo);
                if (BlobStoreUtils.downloadMissingBlob(this.conf, this, key, nimbusSet)) {
                    LOG.debug("Updating blobs state");
                    BlobStoreUtils.createStateInZookeeper(this.conf, key, this.nimbusInfo);
                    checkBlobDownload = true;
                }
            }
        }
        catch (KeyNotFoundException e) {
            throw e;
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        return checkBlobDownload;
    }

    public synchronized void checkForBlobUpdate(String key) {
        BlobStoreUtils.updateKeyForBlobStore(this.conf, this, this.zkClient, key, this.nimbusInfo);
    }

    public void fullCleanup(long age) throws IOException {
        this.fbs.fullCleanup(age);
    }

    @VisibleForTesting
    File getKeyDataDir(String key) {
        return this.fbs.getKeyDir(DATA_PREFIX + key);
    }
}

