/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.security.authentication.token;

import com.google.common.collect.ImmutableMap;
import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.security.Principal;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import javax.jcr.AccessDeniedException;
import javax.jcr.Credentials;
import javax.jcr.RepositoryException;
import org.apache.jackrabbit.api.security.authentication.token.TokenCredentials;
import org.apache.jackrabbit.api.security.user.Authorizable;
import org.apache.jackrabbit.api.security.user.User;
import org.apache.jackrabbit.api.security.user.UserManager;
import org.apache.jackrabbit.oak.api.CommitFailedException;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Root;
import org.apache.jackrabbit.oak.api.Tree;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.namepath.NamePathMapper;
import org.apache.jackrabbit.oak.plugins.identifier.IdentifierManager;
import org.apache.jackrabbit.oak.plugins.tree.TreeUtil;
import org.apache.jackrabbit.oak.security.authentication.token.CommitMarker;
import org.apache.jackrabbit.oak.spi.namespace.NamespaceConstants;
import org.apache.jackrabbit.oak.spi.security.ConfigurationParameters;
import org.apache.jackrabbit.oak.spi.security.authentication.ImpersonationCredentials;
import org.apache.jackrabbit.oak.spi.security.authentication.credentials.CredentialsSupport;
import org.apache.jackrabbit.oak.spi.security.authentication.credentials.SimpleCredentialsSupport;
import org.apache.jackrabbit.oak.spi.security.authentication.token.TokenConstants;
import org.apache.jackrabbit.oak.spi.security.authentication.token.TokenInfo;
import org.apache.jackrabbit.oak.spi.security.authentication.token.TokenProvider;
import org.apache.jackrabbit.oak.spi.security.user.UserConfiguration;
import org.apache.jackrabbit.oak.spi.security.user.util.PasswordUtil;
import org.apache.jackrabbit.util.ISO8601;
import org.apache.jackrabbit.util.Text;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class TokenProviderImpl
implements TokenProvider,
TokenConstants {
    private static final Logger log = LoggerFactory.getLogger(TokenProviderImpl.class);
    static final String PARAM_TOKEN_CLEANUP_THRESHOLD = "tokenCleanupThreshold";
    static final long NO_TOKEN_CLEANUP = 0L;
    static final long DEFAULT_TOKEN_EXPIRATION = 0x6DDD00L;
    static final int DEFAULT_KEY_SIZE = 8;
    private static final char DELIM = '_';
    private final Root root;
    private final ConfigurationParameters options;
    private final CredentialsSupport credentialsSupport;
    private final long tokenExpiration;
    private final UserManager userManager;
    private final IdentifierManager identifierManager;
    private final long cleanupThreshold;

    TokenProviderImpl(@NotNull Root root, @NotNull ConfigurationParameters options, @NotNull UserConfiguration userConfiguration) {
        this(root, options, userConfiguration, SimpleCredentialsSupport.getInstance());
    }

    TokenProviderImpl(@NotNull Root root, @NotNull ConfigurationParameters options, @NotNull UserConfiguration userConfiguration, @NotNull CredentialsSupport credentialsSupport) {
        this.root = root;
        this.options = options;
        this.credentialsSupport = credentialsSupport;
        this.tokenExpiration = options.getConfigValue("tokenExpiration", 0x6DDD00L);
        this.userManager = userConfiguration.getUserManager(root, NamePathMapper.DEFAULT);
        this.identifierManager = new IdentifierManager(root);
        this.cleanupThreshold = options.getConfigValue(PARAM_TOKEN_CLEANUP_THRESHOLD, 0L);
    }

    @Override
    public boolean doCreateToken(@NotNull Credentials credentials) {
        Credentials creds = this.extractCredentials(credentials);
        if (creds == null) {
            return false;
        }
        Object attr = this.credentialsSupport.getAttributes(creds).get(".token");
        return attr != null && attr.toString().isEmpty();
    }

    @Override
    @Nullable
    public TokenInfo createToken(@NotNull Credentials credentials) {
        Map<String, ?> attributes;
        Credentials creds = this.extractCredentials(credentials);
        String uid = creds != null ? this.credentialsSupport.getUserId(creds) : null;
        TokenInfo tokenInfo = null;
        if (uid != null && (tokenInfo = this.createToken(uid, attributes = this.credentialsSupport.getAttributes(creds))) != null && !this.credentialsSupport.setAttributes(creds, (Map<String, ?>)ImmutableMap.of((Object)".token", (Object)tokenInfo.getToken()))) {
            log.debug("Cannot set token attribute to {}", (Object)creds);
        }
        return tokenInfo;
    }

    @Override
    @Nullable
    public TokenInfo createToken(@NotNull String userId, @NotNull Map<String, ?> attributes) {
        Tree tokenParent;
        String error = "Failed to create login token. {}";
        User user = this.getUser(userId);
        Tree tree = tokenParent = user == null ? null : this.getTokenParent(user);
        if (tokenParent != null) {
            try {
                TokenInfo tokenInfo;
                String id = user.getID();
                long creationTime = System.currentTimeMillis();
                long exp = attributes.containsKey("tokenExpiration") ? Long.parseLong(attributes.get("tokenExpiration").toString()) : this.tokenExpiration;
                long expTime = TokenProviderImpl.createExpirationTime(creationTime, exp);
                String uuid = UUID.randomUUID().toString();
                try {
                    String tokenName = uuid;
                    tokenInfo = this.createTokenNode(tokenParent, tokenName, expTime, uuid, id, attributes);
                    this.root.commit(CommitMarker.asCommitAttributes());
                }
                catch (CommitFailedException e) {
                    log.debug("Failed to create token node. Using random name as fallback.");
                    this.root.refresh();
                    tokenInfo = this.createTokenNode(tokenParent, UUID.randomUUID().toString(), expTime, uuid, id, attributes);
                    this.root.commit(CommitMarker.asCommitAttributes());
                }
                this.cleanupExpired(userId, tokenParent, creationTime, tokenInfo.getToken());
                return tokenInfo;
            }
            catch (UnsupportedEncodingException | NoSuchAlgorithmException | RepositoryException | CommitFailedException e) {
                log.error(error, (Object)e.getMessage());
            }
        } else {
            log.error("Unable to get/create token store for user {}.", (Object)userId);
        }
        return null;
    }

    @Override
    @Nullable
    public TokenInfo getTokenInfo(@NotNull String token) {
        int pos = token.indexOf(95);
        String nodeId = pos == -1 ? token : token.substring(0, pos);
        Tree tokenTree = this.identifierManager.getTree(nodeId);
        if (TokenProviderImpl.isValidTokenTree(tokenTree)) {
            try {
                User user = this.getUser(tokenTree);
                if (user != null) {
                    return new TokenInfoImpl(tokenTree, token, user.getID(), user.getPrincipal());
                }
            }
            catch (RepositoryException e) {
                log.debug("Cannot determine userID/principal from token: {}", (Object)e.getMessage());
            }
        }
        return null;
    }

    private static long createExpirationTime(long creationTime, long tokenExpiration) {
        return creationTime + tokenExpiration;
    }

    private static long getExpirationTime(@NotNull Tree tokenTree, long defaultValue) {
        return TreeUtil.getLong(tokenTree, "rep:token.exp", defaultValue);
    }

    private static boolean isExpired(long expirationTime, long loginTime) {
        return expirationTime < loginTime;
    }

    private static void setExpirationTime(@NotNull Tree tree, long time) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(time);
        tree.setProperty("rep:token.exp", ISO8601.format(calendar), Type.DATE);
    }

    @Nullable
    private Credentials extractCredentials(@NotNull Credentials credentials) {
        Credentials creds = credentials;
        if (credentials instanceof ImpersonationCredentials) {
            creds = ((ImpersonationCredentials)credentials).getBaseCredentials();
        }
        if (this.credentialsSupport.getCredentialClasses().contains(creds.getClass())) {
            return creds;
        }
        return null;
    }

    @NotNull
    private static String generateKey(int size) {
        SecureRandom random = new SecureRandom();
        byte[] key = new byte[size];
        random.nextBytes(key);
        StringBuilder res = new StringBuilder(key.length * 2);
        for (byte b : key) {
            res.append(Text.hexTable[b >> 4 & 0xF]);
            res.append(Text.hexTable[b & 0xF]);
        }
        return res.toString();
    }

    @NotNull
    private static String getKeyValue(@NotNull String key, @NotNull String userId) {
        return key + userId;
    }

    private static boolean isValidTokenTree(@Nullable Tree tokenTree) {
        if (tokenTree == null || !tokenTree.exists()) {
            return false;
        }
        return ".tokens".equals(tokenTree.getParent().getName()) && "rep:Token".equals(TreeUtil.getPrimaryTypeName(tokenTree));
    }

    @NotNull
    private Tree getTokenTree(@NotNull TokenInfoImpl tokenInfo) {
        return this.root.getTree(tokenInfo.tokenPath);
    }

    @Nullable
    private User getUser(@NotNull Tree tokenTree) throws RepositoryException {
        String userPath = Text.getRelativeParent(tokenTree.getPath(), 2);
        Authorizable authorizable = this.userManager.getAuthorizableByPath(userPath);
        if (authorizable != null && !authorizable.isGroup() && !((User)authorizable).isDisabled()) {
            return (User)authorizable;
        }
        return null;
    }

    @Nullable
    private User getUser(@NotNull String userId) {
        try {
            Authorizable user = this.userManager.getAuthorizable(userId);
            if (user != null && !user.isGroup()) {
                return (User)user;
            }
            log.debug("Cannot create login token: No corresponding node for User {}.", (Object)userId);
        }
        catch (RepositoryException e) {
            log.debug("Error while accessing user " + userId + '.', (Throwable)e);
        }
        return null;
    }

    @Nullable
    private Tree getTokenParent(@NotNull User user) {
        Tree tokenParent = null;
        String parentPath = null;
        try {
            String userPath = user.getPath();
            parentPath = userPath + '/' + ".tokens";
            Tree userNode = this.root.getTree(userPath);
            tokenParent = TreeUtil.getOrAddChild(userNode, ".tokens", "rep:Unstructured");
            this.root.commit();
        }
        catch (RepositoryException e) {
            log.debug("Error while creating token node {}", (Object)e.getMessage());
        }
        catch (CommitFailedException e) {
            log.debug("Conflict while creating token store -> retrying {}", (Object)e.getMessage());
            this.root.refresh();
            Tree parentTree = this.root.getTree(parentPath);
            tokenParent = parentTree.exists() ? parentTree : null;
        }
        return tokenParent;
    }

    @NotNull
    private TokenInfo createTokenNode(@NotNull Tree parent, @NotNull String tokenName, long expTime, @NotNull String uuid, @NotNull String id, Map<String, ?> attributes) throws AccessDeniedException, UnsupportedEncodingException, NoSuchAlgorithmException {
        Tree tokenNode = TreeUtil.addChild(parent, tokenName, "rep:Token");
        tokenNode.setProperty("jcr:uuid", uuid);
        String key = TokenProviderImpl.generateKey(this.options.getConfigValue("tokenLength", 8));
        String nodeId = IdentifierManager.getIdentifier(tokenNode);
        String token = nodeId + '_' + key;
        String keyHash = PasswordUtil.buildPasswordHash(TokenProviderImpl.getKeyValue(key, id), this.options);
        tokenNode.setProperty("rep:token.key", keyHash);
        TokenProviderImpl.setExpirationTime(tokenNode, expTime);
        attributes.forEach((name, value) -> {
            if (!RESERVED_ATTRIBUTES.contains(name)) {
                tokenNode.setProperty((String)name, value.toString());
            }
        });
        return new TokenInfoImpl(tokenNode, token, id, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void cleanupExpired(@NotNull String userId, @NotNull Tree parent, long currentTime, @NotNull String token) {
        if (this.cleanupThreshold > 0L && TokenProviderImpl.shouldRunCleanup(token)) {
            long expired;
            long active;
            long start;
            block11: {
                start = System.currentTimeMillis();
                active = 0L;
                expired = 0L;
                try {
                    if (parent.getChildrenCount(this.cleanupThreshold) >= this.cleanupThreshold) {
                        for (Tree child : parent.getChildren()) {
                            if (TokenProviderImpl.isExpired(TokenProviderImpl.getExpirationTime(child, Long.MIN_VALUE), currentTime)) {
                                ++expired;
                                child.remove();
                                continue;
                            }
                            ++active;
                        }
                    }
                    if (!this.root.hasPendingChanges()) break block11;
                    this.root.commit(CommitMarker.asCommitAttributes());
                }
                catch (CommitFailedException e) {
                    try {
                        log.debug("Failed to cleanup expired token nodes", (Throwable)e);
                        this.root.refresh();
                    }
                    catch (Throwable throwable) {
                        if (log.isDebugEnabled() && active + expired > 0L) {
                            log.debug("Token cleanup completed in {} ms: removed {}/{} tokens for {}.", new Object[]{System.currentTimeMillis() - start, expired, active + expired, userId});
                        }
                        throw throwable;
                    }
                    if (log.isDebugEnabled() && active + expired > 0L) {
                        log.debug("Token cleanup completed in {} ms: removed {}/{} tokens for {}.", new Object[]{System.currentTimeMillis() - start, expired, active + expired, userId});
                    }
                }
            }
            if (log.isDebugEnabled() && active + expired > 0L) {
                log.debug("Token cleanup completed in {} ms: removed {}/{} tokens for {}.", new Object[]{System.currentTimeMillis() - start, expired, active + expired, userId});
            }
        }
    }

    static boolean shouldRunCleanup(@NotNull String token) {
        return token.charAt(0) < '2';
    }

    final class TokenInfoImpl
    implements TokenInfo {
        private final String token;
        private final String tokenPath;
        private final String userId;
        private final Principal principal;
        private final long expirationTime;
        private final String key;
        private final Map<String, String> mandatoryAttributes;
        private final Map<String, String> publicAttributes;

        private TokenInfoImpl(@NotNull Tree tokenTree, @NotNull String token, @Nullable String userId, Principal principal) {
            this.token = token;
            this.tokenPath = tokenTree.getPath();
            this.userId = userId;
            this.principal = principal;
            this.expirationTime = TokenProviderImpl.getExpirationTime(tokenTree, Long.MIN_VALUE);
            this.key = TreeUtil.getString(tokenTree, "rep:token.key");
            this.mandatoryAttributes = new HashMap<String, String>();
            this.publicAttributes = new HashMap<String, String>();
            for (PropertyState propertyState : tokenTree.getProperties()) {
                String name = propertyState.getName();
                String value = propertyState.getValue(Type.STRING);
                if (TokenConstants.RESERVED_ATTRIBUTES.contains(name)) continue;
                if (this.isMandatoryAttribute(name)) {
                    this.mandatoryAttributes.put(name, value);
                    continue;
                }
                if (!this.isInfoAttribute(name)) continue;
                this.publicAttributes.put(name, value);
            }
        }

        @Nullable
        Principal getPrincipal() {
            return this.principal;
        }

        @Override
        @NotNull
        public String getUserId() {
            return this.userId;
        }

        @Override
        @NotNull
        public String getToken() {
            return this.token;
        }

        @Override
        public boolean isExpired(long loginTime) {
            return TokenProviderImpl.isExpired(this.expirationTime, loginTime);
        }

        @Override
        public boolean resetExpiration(long loginTime) {
            Tree tokenTree;
            if (TokenProviderImpl.this.options.getConfigValue("tokenRefresh", true).booleanValue() && (tokenTree = TokenProviderImpl.this.getTokenTree(this)).exists()) {
                if (this.isExpired(loginTime)) {
                    log.debug("Attempt to reset an expired token.");
                    return false;
                }
                if (this.expirationTime - loginTime <= TokenProviderImpl.this.tokenExpiration / 2L) {
                    try {
                        long expTime = TokenProviderImpl.createExpirationTime(loginTime, TokenProviderImpl.this.tokenExpiration);
                        TokenProviderImpl.setExpirationTime(tokenTree, expTime);
                        TokenProviderImpl.this.root.commit(CommitMarker.asCommitAttributes());
                        log.debug("Successfully reset token expiration time.");
                        return true;
                    }
                    catch (CommitFailedException e) {
                        log.debug("Failed to reset token expiration {}", (Object)e.getMessage());
                        TokenProviderImpl.this.root.refresh();
                    }
                }
            }
            return false;
        }

        @Override
        public boolean remove() {
            Tree tokenTree = TokenProviderImpl.this.getTokenTree(this);
            if (tokenTree.exists()) {
                try {
                    if (tokenTree.remove()) {
                        TokenProviderImpl.this.root.commit(CommitMarker.asCommitAttributes());
                        return true;
                    }
                }
                catch (CommitFailedException e) {
                    log.debug("Error while removing expired token {}", (Object)e.getMessage());
                }
            }
            return false;
        }

        @Override
        public boolean matches(@NotNull TokenCredentials tokenCredentials) {
            String tk = tokenCredentials.getToken();
            int pos = tk.lastIndexOf(95);
            if (pos > -1) {
                tk = tk.substring(pos + 1);
            }
            if (!PasswordUtil.isSame(this.key, TokenProviderImpl.getKeyValue(tk, this.userId))) {
                return false;
            }
            for (Map.Entry<String, String> mandatory : this.mandatoryAttributes.entrySet()) {
                String name2 = mandatory.getKey();
                String expectedValue = mandatory.getValue();
                if (expectedValue.equals(tokenCredentials.getAttribute(name2))) continue;
                return false;
            }
            List<String> attrNames = Arrays.asList(tokenCredentials.getAttributeNames());
            this.publicAttributes.forEach((name, value) -> {
                if (!attrNames.contains(name)) {
                    tokenCredentials.setAttribute(name, value);
                }
            });
            return true;
        }

        @Override
        @NotNull
        public Map<String, String> getPrivateAttributes() {
            return Collections.unmodifiableMap(this.mandatoryAttributes);
        }

        @Override
        @NotNull
        public Map<String, String> getPublicAttributes() {
            return Collections.unmodifiableMap(this.publicAttributes);
        }

        private boolean isMandatoryAttribute(@NotNull String attributeName) {
            return attributeName.startsWith(".token");
        }

        private boolean isInfoAttribute(@NotNull String attributeName) {
            String prefix = Text.getNamespacePrefix(attributeName);
            return !NamespaceConstants.RESERVED_PREFIXES.contains(prefix);
        }
    }
}

