/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.index;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.base.Throwables;
import com.google.common.base.Ticker;
import com.google.common.collect.Maps;
import java.util.Calendar;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import javax.management.openmbean.CompositeDataSupport;
import javax.management.openmbean.CompositeType;
import javax.management.openmbean.OpenDataException;
import javax.management.openmbean.OpenType;
import javax.management.openmbean.SimpleType;
import javax.management.openmbean.TabularData;
import javax.management.openmbean.TabularDataSupport;
import javax.management.openmbean.TabularType;
import org.apache.jackrabbit.oak.plugins.index.CorruptIndexHandler;
import org.apache.jackrabbit.oak.stats.Clock;
import org.apache.jackrabbit.oak.stats.MeterStats;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TrackingCorruptIndexHandler
implements CorruptIndexHandler {
    static final String CORRUPT_INDEX_METER_NAME = "corrupt-index";
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private Clock clock = Clock.SIMPLE;
    private long errorWarnIntervalMillis = TimeUnit.MINUTES.toMillis(15L);
    private long indexerCycleCount;
    private long corruptIntervalMillis = TimeUnit.MINUTES.toMillis(30L);
    private final Map<String, CorruptIndexInfo> indexes = Maps.newConcurrentMap();
    private MeterStats meter;

    void setMeterStats(MeterStats meter) {
        this.meter = meter;
    }

    public Map<String, CorruptIndexInfo> getCorruptIndexData(String asyncName) {
        if (this.corruptIntervalMillis <= 0L) {
            return Collections.emptyMap();
        }
        HashMap result = Maps.newHashMap();
        for (CorruptIndexInfo info : this.indexes.values()) {
            if (!asyncName.equals(info.asyncName) || !info.isFailingSinceLongTime()) continue;
            result.put(info.path, info);
        }
        return result;
    }

    public Map<String, CorruptIndexInfo> getFailingIndexData(String asyncName) {
        HashMap result = Maps.newHashMap();
        for (CorruptIndexInfo info : this.indexes.values()) {
            if (!asyncName.equals(info.asyncName)) continue;
            result.put(info.path, info);
        }
        return result;
    }

    public void markWorkingIndexes(Set<String> updatedIndexPaths) {
        ++this.indexerCycleCount;
        for (String indexPath : updatedIndexPaths) {
            CorruptIndexInfo info = this.indexes.remove(indexPath);
            if (info == null) continue;
            this.log.info("Index at [{}] which was so far failing {} is now working again.", (Object)info.path, (Object)info.getStats());
        }
        if (this.meter != null) {
            this.meter.mark(this.indexes.size());
        }
    }

    public boolean isFailing(String asyncName) {
        return !this.getFailingIndexData(asyncName).isEmpty();
    }

    @Override
    public boolean skippingCorruptIndex(String async, String indexPath, Calendar corruptSince) {
        CorruptIndexInfo info = this.getOrCreateInfo(async, indexPath);
        if (info.skippedIndexing((Calendar)Preconditions.checkNotNull((Object)corruptSince))) {
            this.log.warn("Ignoring corrupt index [{}] which has been marked as corrupt [{}]. This index MUST be reindexed for indexing to work properly", (Object)indexPath, (Object)info.getStats());
            return true;
        }
        return false;
    }

    @Override
    public void indexUpdateFailed(String async, String indexPath, Exception e) {
        if (this.meter != null) {
            this.meter.mark();
        }
        this.getOrCreateInfo(async, indexPath).addFailure(e);
    }

    public void setCorruptInterval(long interval, TimeUnit unit) {
        this.corruptIntervalMillis = unit.toMillis(interval);
    }

    public void setErrorWarnInterval(long errorWarnInterval, TimeUnit unit) {
        this.errorWarnIntervalMillis = unit.toMillis(errorWarnInterval);
    }

    void setClock(Clock clock) {
        this.clock = clock;
    }

    long getCorruptIntervalMillis() {
        return this.corruptIntervalMillis;
    }

    long getErrorWarnIntervalMillis() {
        return this.errorWarnIntervalMillis;
    }

    private long getTime() {
        return this.clock.getTime();
    }

    private synchronized CorruptIndexInfo getOrCreateInfo(String asyncName, String indexPath) {
        CorruptIndexInfo info = this.indexes.get(indexPath);
        if (info == null) {
            info = new CorruptIndexInfo(asyncName, indexPath);
            this.indexes.put(indexPath, info);
        }
        return info;
    }

    public TabularData getFailingIndexStats(String asyncName) {
        TabularDataSupport tds;
        try {
            TabularType tt = new TabularType(TrackingCorruptIndexHandler.class.getName(), "Failing Index Stats", FailingIndexStats.TYPE, new String[]{"path"});
            tds = new TabularDataSupport(tt);
            Map<String, CorruptIndexInfo> infos = this.getFailingIndexData(asyncName);
            for (Map.Entry<String, CorruptIndexInfo> e : infos.entrySet()) {
                FailingIndexStats stats = new FailingIndexStats(e.getValue());
                tds.put(stats.toCompositeData());
            }
        }
        catch (OpenDataException e) {
            throw new IllegalStateException(e);
        }
        return tds;
    }

    private static class ClockTicker
    extends Ticker {
        private final Clock clock;

        public ClockTicker(Clock clock) {
            this.clock = clock;
        }

        public long read() {
            return TimeUnit.MILLISECONDS.toNanos(this.clock.getTime());
        }
    }

    private static class FailingIndexStats {
        static final String[] FIELD_NAMES = new String[]{"path", "stats", "markedCorrupt", "failingSince", "exception"};
        static final String[] FIELD_DESCRIPTIONS = new String[]{"Path", "Failure stats", "Marked as corrupt", "Failure start time", "Exception"};
        static final OpenType[] FIELD_TYPES = new OpenType[]{SimpleType.STRING, SimpleType.STRING, SimpleType.BOOLEAN, SimpleType.STRING, SimpleType.STRING};
        static final CompositeType TYPE = FailingIndexStats.createCompositeType();
        private final CorruptIndexInfo info;

        static CompositeType createCompositeType() {
            try {
                return new CompositeType(FailingIndexStats.class.getName(), "Composite data type for Failing Index statistics", FIELD_NAMES, FIELD_DESCRIPTIONS, FIELD_TYPES);
            }
            catch (OpenDataException e) {
                throw new IllegalStateException(e);
            }
        }

        public FailingIndexStats(CorruptIndexInfo info) {
            this.info = info;
        }

        CompositeDataSupport toCompositeData() {
            Object[] values = new Object[]{this.info.path, this.info.getStats(), this.info.isMarkedAsCorrupt(), String.format("%tc", this.info.getCorruptSinceAsCal().getTimeInMillis()), this.info.getLastException()};
            try {
                return new CompositeDataSupport(TYPE, FIELD_NAMES, values);
            }
            catch (OpenDataException e) {
                throw new IllegalStateException(e);
            }
        }
    }

    public class CorruptIndexInfo {
        private final String asyncName;
        private final String path;
        private final long lastIndexerCycleCount;
        private final Stopwatch watch;
        private String exception;
        private int failureCount;
        private int skippedCount;
        private long corruptSince;

        CorruptIndexInfo(String asyncName, String path) {
            this.lastIndexerCycleCount = TrackingCorruptIndexHandler.this.indexerCycleCount;
            this.watch = Stopwatch.createStarted((Ticker)new ClockTicker(TrackingCorruptIndexHandler.this.clock));
            this.exception = "";
            this.asyncName = asyncName;
            this.path = path;
            this.corruptSince = TrackingCorruptIndexHandler.this.getTime();
        }

        void addFailure(Exception e) {
            this.exception = Throwables.getStackTraceAsString((Throwable)e);
            ++this.failureCount;
        }

        boolean skippedIndexing(Calendar corruptSince) {
            ++this.skippedCount;
            this.corruptSince = corruptSince.getTimeInMillis();
            if (this.watch.elapsed(TimeUnit.MILLISECONDS) > TrackingCorruptIndexHandler.this.errorWarnIntervalMillis) {
                this.watch.reset().start();
                return true;
            }
            return false;
        }

        public boolean isFailingSinceLongTime() {
            return TrackingCorruptIndexHandler.this.getTime() - this.corruptSince > TrackingCorruptIndexHandler.this.corruptIntervalMillis;
        }

        public String getStats() {
            return String.format("since %tc ,%d indexing cycles, failed %d times, skipped %d time", this.corruptSince, this.getCycleCount(), this.failureCount, this.skippedCount);
        }

        public Calendar getCorruptSinceAsCal() {
            Calendar cal = Calendar.getInstance();
            cal.setTimeInMillis(this.corruptSince);
            return cal;
        }

        public String getLastException() {
            return this.exception;
        }

        public boolean isMarkedAsCorrupt() {
            return this.skippedCount > 0;
        }

        public int getSkippedCount() {
            return this.skippedCount;
        }

        public String getPath() {
            return this.path;
        }

        private long getCycleCount() {
            return TrackingCorruptIndexHandler.this.indexerCycleCount - this.lastIndexerCycleCount;
        }
    }
}

