/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.reef.tang;

import org.apache.reef.tang.annotations.Name;
import org.apache.reef.tang.annotations.NamedParameter;
import org.apache.reef.tang.annotations.Parameter;
import org.apache.reef.tang.exceptions.BindException;
import org.apache.reef.tang.exceptions.InjectionException;
import org.apache.reef.tang.formats.AvroConfigurationSerializer;
import org.apache.reef.tang.formats.ConfigurationSerializer;
import org.junit.Assert;
import org.junit.Test;

import javax.inject.Inject;
import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

public class TestSetInjection {
  @Test
  public void testStringInjectDefault() throws InjectionException {
    final Set<String> actual = Tang.Factory.getTang().newInjector().getInstance(Box.class).getStrings();

    final Set<String> expected = new HashSet<>();
    expected.add("one");
    expected.add("two");
    expected.add("three");

    Assert.assertEquals(expected, actual);
  }

  @Test
  public void testObjectInjectDefault() throws InjectionException, BindException {
    final Injector i = Tang.Factory.getTang().newInjector();
    i.bindVolatileInstance(Integer.class, 42);
    i.bindVolatileInstance(Float.class, 42.0001f);
    final Set<Number> actual = i.getInstance(Pool.class).getNumbers();
    final Set<Number> expected = new HashSet<>();
    expected.add(42);
    expected.add(42.0001f);
    Assert.assertEquals(expected, actual);
  }

  @Test
  public void testStringInjectBound() throws InjectionException, BindException {
    final JavaConfigurationBuilder cb = Tang.Factory.getTang().newConfigurationBuilder();
    cb.bindSetEntry(SetOfStrings.class, "four");
    cb.bindSetEntry(SetOfStrings.class, "five");
    cb.bindSetEntry(SetOfStrings.class, "six");
    final Set<String> actual = Tang.Factory.getTang().newInjector(cb.build()).getInstance(Box.class).getStrings();

    final Set<String> expected = new HashSet<>();
    expected.add("four");
    expected.add("five");
    expected.add("six");

    Assert.assertEquals(expected, actual);
  }

  @Test
  public void testObjectInjectBound() throws InjectionException, BindException {
    final JavaConfigurationBuilder cb = Tang.Factory.getTang().newConfigurationBuilder();
    cb.bindSetEntry(SetOfClasses.class, Short.class);
    cb.bindSetEntry(SetOfClasses.class, Float.class);

    final Injector i = Tang.Factory.getTang().newInjector(cb.build());
    i.bindVolatileInstance(Short.class, (short) 4);
    i.bindVolatileInstance(Float.class, 42.0001f);
    final Set<Number> actual = i.getInstance(Pool.class).getNumbers();
    final Set<Number> expected = new HashSet<>();
    expected.add((short) 4);
    expected.add(42.0001f);
    Assert.assertEquals(expected, actual);
  }

  @Test
  public void testStringInjectRoundTrip() throws InjectionException, BindException, IOException {
    final JavaConfigurationBuilder cb = Tang.Factory.getTang().newConfigurationBuilder();
    cb.bindSetEntry(SetOfStrings.class, "four");
    cb.bindSetEntry(SetOfStrings.class, "five");
    cb.bindSetEntry(SetOfStrings.class, "six");

    final ConfigurationSerializer serializer = new AvroConfigurationSerializer();

    final String s = serializer.toString(cb.build());
    final JavaConfigurationBuilder cb2 = Tang.Factory.getTang().newConfigurationBuilder();
    final Configuration conf = serializer.fromString(s);
    cb2.addConfiguration(conf);

    final Set<String> actual = Tang.Factory.getTang().newInjector(cb2.build()).getInstance(Box.class).getStrings();

    final Set<String> expected = new HashSet<>();
    expected.add("four");
    expected.add("five");
    expected.add("six");

    Assert.assertEquals(expected, actual);
  }

  @Test
  public void testObjectInjectRoundTrip() throws InjectionException, BindException, IOException {
    final JavaConfigurationBuilder cb = Tang.Factory.getTang().newConfigurationBuilder();
    cb.bindSetEntry(SetOfClasses.class, Short.class);
    cb.bindSetEntry(SetOfClasses.class, Float.class);

    final ConfigurationSerializer serializer = new AvroConfigurationSerializer();

    final String s = serializer.toString(cb.build());
    final JavaConfigurationBuilder cb2 = Tang.Factory.getTang().newConfigurationBuilder();
    final Configuration conf = serializer.fromString(s);
    cb2.addConfiguration(conf);

    final Injector i = Tang.Factory.getTang().newInjector(cb2.build());
    i.bindVolatileInstance(Short.class, (short) 4);
    i.bindVolatileInstance(Float.class, 42.0001f);
    final Set<Number> actual = i.getInstance(Pool.class).getNumbers();
    final Set<Number> expected = new HashSet<>();
    expected.add((short) 4);
    expected.add(42.0001f);
    Assert.assertEquals(expected, actual);
  }

  @Test
  public void testDefaultAsClass() throws InjectionException, BindException {
    final Injector i = Tang.Factory.getTang().newInjector();
    i.bindVolatileInstance(Integer.class, 1);
    i.bindVolatileInstance(Float.class, 2f);
    final Set<Number> actual = i.getNamedInstance(SetOfClassesDefaultClass.class);
    final Set<Number> expected = new HashSet<>();
    expected.add(1);
    Assert.assertEquals(expected, actual);
  }

}

@NamedParameter(default_values = {"one", "two", "three"})
class SetOfStrings implements Name<Set<String>> {
}

class Box {
  private final Set<String> strings;

  public Set<String> getStrings() {
    return strings;
  }

  @Inject
  Box(@Parameter(SetOfStrings.class) final Set<String> strings) {
    this.strings = strings;
  }
}

@NamedParameter(default_values = {"java.lang.Integer", "java.lang.Float"})
class SetOfClasses implements Name<Set<Number>> {
}

class Pool {
  private final Set<Number> numbers;

  public Set<Number> getNumbers() {
    return numbers;
  }

  @Inject
  Pool(@Parameter(SetOfClasses.class) final Set<Number> numbers) {
    this.numbers = numbers;
  }
}

@NamedParameter(default_class = Integer.class)
class SetOfClassesDefaultClass implements Name<Set<Number>> {
}
