/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.reef.tang.implementation.types;

import org.apache.reef.tang.types.Node;
import org.apache.reef.tang.util.MonotonicTreeMap;

import java.util.Collection;
import java.util.Map;

public abstract class AbstractNode implements Node {
  protected final Map<String, Node> children = new MonotonicTreeMap<>();
  private final Node parent;
  private final String name;
  private final String fullName;

  public AbstractNode(final Node parent, final String name, final String fullName) {
    this.parent = parent;
    this.name = name;
    this.fullName = fullName;
    if (parent != null) {
      if (name.length() == 0) {
        throw new IllegalArgumentException(
            "Zero length child name means bad news");
      }
      parent.put(this);
    }
  }

  @Override
  public Collection<Node> getChildren() {
    return children.values();
  }

  @Override
  public String getFullName() {
    return fullName;
  }

  @Override
  public boolean equals(final Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    final AbstractNode n = (AbstractNode) o;
    final boolean parentsEqual;
    if (n.parent == this.parent) {
      parentsEqual = true;
    } else if (n.parent == null) {
      parentsEqual = false;
    } else if (this.parent == null) {
      parentsEqual = false;
    } else {
      parentsEqual = n.parent.equals(this.parent);
    }
    if (!parentsEqual) {
      return false;
    }
    return this.name.equals(n.name);
  }

  @Override
  public int hashCode() {
    return getFullName().hashCode();
  }

  @Override
  public Node getParent() {
    return parent;
  }

  @Override
  public boolean contains(final String key) {
    return children.containsKey(key);
  }

  @Override
  public Node get(final String key) {
    return children.get(key);
  }

  @Override
  public void put(final Node n) {
    children.put(n.getName(), n);
  }

  @SuppressWarnings("unused")
  private String toIndentedString(final int level) {
    final StringBuilder sb = new StringBuilder();
    for (int i = 0; i < level; i++) {
      sb.append("\t");
    }
    sb.append(toString() + "\n");
    if (children != null) {
      for (final Node n : children.values()) {
        sb.append(((AbstractNode) n).toIndentedString(level + 1));
      }
    }
    return sb.toString();
  }

  @Override
  public String toString() {
    return "[" + this.getClass().getSimpleName() + " '" + getFullName() + "']";
  }

  @Override
  public String getName() {
    return name;
  }

  @Override
  public int compareTo(final Node n) {
    return getFullName().compareTo(n.getFullName());
  }
}
