﻿// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
// 
//   http://www.apache.org/licenses/LICENSE-2.0
// 
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

using System;
using System.Runtime.CompilerServices;
using System.Threading;

namespace Org.Apache.REEF.Utilities.AsyncUtils
{
    /// <summary>
    /// This is to avoid deadlocks in clients that use synchronization context (WinForms/ASP.NET)
    /// Use at asynchronous entry points of all async REEF client APIs.
    /// 
    /// There is no need to use this in evaluator/driver.
    /// </summary>
    public struct RemoveSynchronizationContextAwaiter : INotifyCompletion
    {
        public bool IsCompleted
        {
            get { return SynchronizationContext.Current == null; }
        }

        public void OnCompleted(Action continuation)
        {
            SynchronizationContext previousSynchronizationContext = SynchronizationContext.Current;
            try
            {
                SynchronizationContext.SetSynchronizationContext(null);
                continuation();
            }
            finally
            {
                SynchronizationContext.SetSynchronizationContext(previousSynchronizationContext);
            }
        }

        public RemoveSynchronizationContextAwaiter GetAwaiter()
        {
            return this;
        }

        public void GetResult()
        {
        }
    }
}