/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jclouds.openstack.glance.v1_0.features;

import static org.jclouds.openstack.glance.v1_0.options.CreateImageOptions.Builder.copyFrom;
import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertTrue;

import javax.ws.rs.core.MediaType;

import org.jclouds.http.HttpRequest;
import org.jclouds.http.HttpResponse;
import org.jclouds.io.Payloads;
import org.jclouds.io.payloads.StringPayload;
import org.jclouds.openstack.glance.v1_0.GlanceApi;
import org.jclouds.openstack.glance.v1_0.functions.ParseImageDetailsFromHeadersTest;
import org.jclouds.openstack.glance.v1_0.internal.BaseGlanceExpectTest;
import org.jclouds.openstack.glance.v1_0.options.UpdateImageOptions;
import org.jclouds.openstack.glance.v1_0.parse.ParseImageDetailsTest;
import org.jclouds.openstack.glance.v1_0.parse.ParseImagesInDetailTest;
import org.jclouds.openstack.glance.v1_0.parse.ParseImagesTest;
import org.jclouds.util.Strings2;
import org.testng.annotations.Test;

import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.ImmutableSet;

@Test(groups = "unit", testName = "ImageApiExpectTest")
public class ImageApiExpectTest extends BaseGlanceExpectTest {

   public void testListWhenResponseIs2xx() throws Exception {
      HttpRequest list = HttpRequest.builder().method("GET")
            .endpoint("https://glance.jclouds.org:9292/v1.0/images")
            .addHeader("Accept", "application/json")
            .addHeader("X-Auth-Token", authToken).build();

      HttpResponse listResponse = HttpResponse.builder().statusCode(200)
            .payload(payloadFromResource("/images.json")).build();

      GlanceApi apiWhenExist = requestsSendResponses(keystoneAuthWithUsernameAndPassword,
            responseWithKeystoneAccess, versionNegotiationRequest, versionNegotiationResponse,
            list, listResponse);

      assertEquals(apiWhenExist.getConfiguredRegions(), ImmutableSet.of("az-1.region-a.geo-1"));

      assertEquals(apiWhenExist.getImageApi("az-1.region-a.geo-1").list().concat().toString(),
            new ParseImagesTest().expected().toString());
   }

   public void testListInDetailWhenResponseIs2xx() throws Exception {
      HttpRequest listInDetail = HttpRequest.builder().method("GET")
            .endpoint("https://glance.jclouds.org:9292/v1.0/images/detail")
            .addHeader("Accept", "application/json")
            .addHeader("X-Auth-Token", authToken).build();

      HttpResponse listInDetailResponse = HttpResponse.builder().statusCode(200)
            .payload(payloadFromResource("/images_detail.json")).build();

      GlanceApi apiWhenExistInDetail = requestsSendResponses(keystoneAuthWithUsernameAndPassword,
            responseWithKeystoneAccess, versionNegotiationRequest, versionNegotiationResponse,
            listInDetail, listInDetailResponse);

      assertEquals(apiWhenExistInDetail.getConfiguredRegions(), ImmutableSet.of("az-1.region-a.geo-1"));

      assertEquals(apiWhenExistInDetail.getImageApi("az-1.region-a.geo-1").listInDetail().concat().toString(),
            new ParseImagesInDetailTest().expected().toString());
   }

   public void testShowWhenResponseIs2xx() throws Exception {
      HttpRequest show = HttpRequest.builder().method("HEAD")
            .endpoint("https://glance.jclouds.org:9292/v1.0/images/fcc451d0-f6e4-4824-ad8f-70ec12326d07")
            .addHeader("Accept", MediaType.APPLICATION_JSON)
            .addHeader("X-Auth-Token", authToken).build();

      HttpResponse showResponse = new ParseImageDetailsFromHeadersTest().response;

      GlanceApi apiWhenExist = requestsSendResponses(keystoneAuthWithUsernameAndPassword,
            responseWithKeystoneAccess, versionNegotiationRequest, versionNegotiationResponse,
            show, showResponse);

      assertEquals(apiWhenExist.getConfiguredRegions(), ImmutableSet.of("az-1.region-a.geo-1"));

      assertEquals(apiWhenExist.getImageApi("az-1.region-a.geo-1").get("fcc451d0-f6e4-4824-ad8f-70ec12326d07").toString(),
            new ParseImageDetailsFromHeadersTest().expected().toString());
   }

   public void testGetAsStreamWhenResponseIs2xx() throws Exception {
      HttpRequest get = HttpRequest.builder().method("GET")
            .endpoint("https://glance.jclouds.org:9292/v1.0/images/fcc451d0-f6e4-4824-ad8f-70ec12326d07")
            .addHeader("Accept", MediaType.APPLICATION_JSON)
            .addHeader("X-Auth-Token", authToken).build();

      HttpResponse getResponse = HttpResponse.builder().statusCode(200).payload(Payloads.newStringPayload("foo")).build();

      GlanceApi apiWhenExist = requestsSendResponses(keystoneAuthWithUsernameAndPassword,
            responseWithKeystoneAccess, versionNegotiationRequest, versionNegotiationResponse,
            get, getResponse);

      assertEquals(apiWhenExist.getConfiguredRegions(), ImmutableSet.of("az-1.region-a.geo-1"));

      assertEquals(Strings2.toStringAndClose(apiWhenExist.getImageApi("az-1.region-a.geo-1").getAsStream("fcc451d0-f6e4-4824-ad8f-70ec12326d07")),
               "foo");
   }

   public void testCreateWhenResponseIs2xx() throws Exception {
      HttpRequest get = HttpRequest.builder().method("POST")
            .endpoint("https://glance.jclouds.org:9292/v1.0/images")
            .addHeader("x-image-meta-name", "test")
            .addHeader("Accept", MediaType.APPLICATION_JSON)
            .addHeader("X-Auth-Token", authToken)
            .payload(payloadFromStringWithContentType("somedata", MediaType.APPLICATION_OCTET_STREAM)).build();

      HttpResponse createResponse = HttpResponse.builder().statusCode(200)
            .payload(payloadFromResource("/image.json")).build();

      GlanceApi apiWhenExist = requestsSendResponses(keystoneAuthWithUsernameAndPassword,
            responseWithKeystoneAccess, versionNegotiationRequest, versionNegotiationResponse,
            get, createResponse);

      assertEquals(apiWhenExist.getConfiguredRegions(), ImmutableSet.of("az-1.region-a.geo-1"));

      assertEquals(apiWhenExist.getImageApi("az-1.region-a.geo-1").create("test", new StringPayload("somedata")),
            new ParseImageDetailsTest().expected());
   }

   public void testImageCreateCopyFrom() throws Exception {
      HttpRequest get = HttpRequest.builder().method("POST")
            .endpoint("https://glance.jclouds.org:9292/v1.0/images")
            .addHeader("x-image-meta-name", "test")
            .addHeader("Accept", MediaType.APPLICATION_JSON)
            .addHeader("x-glance-api-copy-from", "http://1.1.1.1/Installs/Templates/tiny/tinylinux-v2.qcow2")
            .addHeader("X-Auth-Token", authToken)
            .payload(payloadFromStringWithContentType("", MediaType.APPLICATION_OCTET_STREAM)).build();

      HttpResponse createResponse = HttpResponse.builder().statusCode(200)
            .payload(payloadFromResource("/image.json")).build();

      GlanceApi apiWhenExist = requestsSendResponses(keystoneAuthWithUsernameAndPassword,
            responseWithKeystoneAccess, versionNegotiationRequest, versionNegotiationResponse,
            get, createResponse);

      assertEquals(apiWhenExist.getConfiguredRegions(), ImmutableSet.of("az-1.region-a.geo-1"));

      assertEquals(apiWhenExist.getImageApi("az-1.region-a.geo-1").create("test", new StringPayload(""), copyFrom("http://1.1.1.1/Installs/Templates/tiny/tinylinux-v2.qcow2")),
            new ParseImageDetailsTest().expected());
}   

   public void testReserveWhenResponseIs2xx() throws Exception {
      HttpRequest get = HttpRequest.builder().method("POST")
            .endpoint("https://glance.jclouds.org:9292/v1.0/images")
            .addHeader("x-image-meta-name", "test")
            .addHeader("Accept", MediaType.APPLICATION_JSON)
            .addHeader("X-Auth-Token", authToken).build();

      HttpResponse createResponse = HttpResponse.builder().statusCode(200)
            .payload(payloadFromResource("/image.json")).build();

      GlanceApi apiWhenExist = requestsSendResponses(keystoneAuthWithUsernameAndPassword,
            responseWithKeystoneAccess, versionNegotiationRequest, versionNegotiationResponse,
            get, createResponse);

      assertEquals(apiWhenExist.getConfiguredRegions(), ImmutableSet.of("az-1.region-a.geo-1"));

      assertEquals(apiWhenExist.getImageApi("az-1.region-a.geo-1").reserve("test"), new ParseImageDetailsTest().expected());
   }

   public void testUpdateMetadataWhenResponseIs2xx() throws Exception {
      HttpRequest get = HttpRequest.builder().method("PUT")
            .endpoint("https://glance.jclouds.org:9292/v1.0/images/fcc451d0-f6e4-4824-ad8f-70ec12326d07")
            .headers(
                  ImmutableMultimap.<String, String>builder()
                        .put("Accept", MediaType.APPLICATION_JSON)
                        .put("X-Image-Meta-Name", "newname")
                        .put("X-Image-Meta-Is_public", "true")
                        .put("X-Image-Meta-Protected", "true")
                        .put("X-Image-Meta-Checksum", "XXXX")
                        .put("X-Image-Meta-Location", "somewhere")
                        .put("X-Image-Meta-Min_disk", "10")
                        .put("X-Image-Meta-Min_ram", "2048")
                        .put("X-Auth-Token", authToken).build())
            .build();

      HttpResponse updateResponse = HttpResponse.builder().statusCode(200)
            .payload(payloadFromResource("/image.json")).build();

      GlanceApi apiWhenExist = requestsSendResponses(keystoneAuthWithUsernameAndPassword,
            responseWithKeystoneAccess, versionNegotiationRequest, versionNegotiationResponse,
            get, updateResponse);

      assertEquals(apiWhenExist.getConfiguredRegions(), ImmutableSet.of("az-1.region-a.geo-1"));

      assertEquals(apiWhenExist.getImageApi("az-1.region-a.geo-1")
            .update("fcc451d0-f6e4-4824-ad8f-70ec12326d07",
                  UpdateImageOptions.Builder.name("newname"),
                  UpdateImageOptions.Builder.isPublic(true),
                  UpdateImageOptions.Builder.isProtected(true),
                  UpdateImageOptions.Builder.checksum("XXXX"),
                  UpdateImageOptions.Builder.location("somewhere"),
                  UpdateImageOptions.Builder.minDisk(10),
                  UpdateImageOptions.Builder.minRam(2048)),
               new ParseImageDetailsTest().expected());
   }

   public void testUpdateImageWhenResponseIs2xx() throws Exception {
      HttpRequest get = HttpRequest.builder().method("PUT")
            .endpoint("https://glance.jclouds.org:9292/v1.0/images/fcc451d0-f6e4-4824-ad8f-70ec12326d07")
            .addHeader("Accept", MediaType.APPLICATION_JSON)
            .addHeader("X-Auth-Token", authToken)
            .payload(payloadFromStringWithContentType("somenewdata", MediaType.APPLICATION_OCTET_STREAM))
            .build();

      HttpResponse updateResponse = HttpResponse.builder().statusCode(200)
            .payload(payloadFromResource("/image.json")).build();


      GlanceApi apiWhenExist = requestsSendResponses(keystoneAuthWithUsernameAndPassword,
            responseWithKeystoneAccess, versionNegotiationRequest, versionNegotiationResponse,
            get, updateResponse);

      assertEquals(apiWhenExist.getConfiguredRegions(), ImmutableSet.of("az-1.region-a.geo-1"));

      assertEquals(apiWhenExist.getImageApi("az-1.region-a.geo-1").upload("fcc451d0-f6e4-4824-ad8f-70ec12326d07",
            new StringPayload("somenewdata")), new ParseImageDetailsTest().expected());
   }

   public void testUpdateNameAndImageWhenResponseIs2xx() throws Exception {
      HttpRequest get = HttpRequest.builder().method("PUT")
            .endpoint("https://glance.jclouds.org:9292/v1.0/images/fcc451d0-f6e4-4824-ad8f-70ec12326d07")
            .headers(
                  ImmutableMultimap.<String, String>builder()
                        .put("Accept", MediaType.APPLICATION_JSON)
                        .put("X-Image-Meta-Name", "anothernewname")
                        .put("X-Auth-Token", authToken).build())
            .payload(payloadFromStringWithContentType("somenewdata", MediaType.APPLICATION_OCTET_STREAM))
            .build();

      HttpResponse updateResponse = HttpResponse.builder().statusCode(200)
            .payload(payloadFromResource("/image.json")).build();


      GlanceApi apiWhenExist = requestsSendResponses(keystoneAuthWithUsernameAndPassword,
            responseWithKeystoneAccess, versionNegotiationRequest, versionNegotiationResponse,
            get, updateResponse);

      assertEquals(apiWhenExist.getConfiguredRegions(), ImmutableSet.of("az-1.region-a.geo-1"));

      assertEquals(apiWhenExist.getImageApi("az-1.region-a.geo-1").upload("fcc451d0-f6e4-4824-ad8f-70ec12326d07",
            new StringPayload("somenewdata"), UpdateImageOptions.Builder.name("anothernewname")), new ParseImageDetailsTest().expected());
   }

   public void testDeleteWhenResponseIs2xx() throws Exception {
      HttpRequest get = HttpRequest.builder().method("DELETE")
            .endpoint("https://glance.jclouds.org:9292/v1.0/images/fcc451d0-f6e4-4824-ad8f-70ec12326d07")
            .addHeader("Accept", MediaType.APPLICATION_JSON)
            .addHeader("X-Auth-Token", authToken).build();

      HttpResponse getResponse = HttpResponse.builder().statusCode(200).build();

      GlanceApi apiWhenExist = requestsSendResponses(keystoneAuthWithUsernameAndPassword,
            responseWithKeystoneAccess, versionNegotiationRequest, versionNegotiationResponse,
            get, getResponse);

      assertEquals(apiWhenExist.getConfiguredRegions(), ImmutableSet.of("az-1.region-a.geo-1"));

      assertTrue(apiWhenExist.getImageApi("az-1.region-a.geo-1").delete("fcc451d0-f6e4-4824-ad8f-70ec12326d07"));
   }
}
