////////////////////////////////////////////////////////////////////////////////
//
//  Licensed to the Apache Software Foundation (ASF) under one or more
//  contributor license agreements.  See the NOTICE file distributed with
//  this work for additional information regarding copyright ownership.
//  The ASF licenses this file to You under the Apache License, Version 2.0
//  (the "License"); you may not use this file except in compliance with
//  the License.  You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
////////////////////////////////////////////////////////////////////////////////
package org.apache.flex.geom
{

	public class Matrix
	{
		public function Matrix(a:Number = 1, b:Number = 0, c:Number = 0, d:Number = 1, tx:Number = 0, ty:Number = 0)
		{
			this.a = a;
			this.b = b;
			this.c = c;
			this.d = d;
			this.tx = tx;
			this.ty = ty;

		}
		public var a:Number;
		public var b:Number;
		public var c:Number;
		public var d:Number;
		public var tx:Number;
		public var ty:Number;

		/**
		 *	Returns a copy of the Matrix
	     *  @langversion 3.0
	     *  @playerversion Flash 10.2
	     *  @playerversion AIR 2.6
	     *  @productversion FlexJS 0.7
		 */
		public function clone():Matrix
		{
			return new Matrix(a, b, c, d, tx, ty);
		}
		
		/**
		 *  Adds the Matrix the current one
		 *  Returns the matrix so the methods can be chained.
	     *  @langversion 3.0
	     *  @playerversion Flash 10.2
	     *  @playerversion AIR 2.6
	     *  @productversion FlexJS 0.7
		 */
		public function concat(m:Matrix):Matrix
		{
			var newa:Number = a * m.a + b * m.c;
			b = a * m.b + b * m.d;
			a = newa;
			
			var newc:Number = c * m.a + d * m.c;
			d = c * m.b + d * m.d;
			c = newc;
			
			var newtx:Number = tx * m.a + ty * m.c + m.tx;
			ty = tx * m.b + ty * m.d + m.ty;
			tx = newtx;
			return this;
		}

		/**
		 *  Calculates the Matrix determinant
	     *  @langversion 3.0
	     *  @playerversion Flash 10.2
	     *  @playerversion AIR 2.6
	     *  @productversion FlexJS 0.7
		 */
		public function determinant():Number
		{
			return a * d - b * c;
		}
		
		/**
		 *  Inverts the Matrix.
		 *  Returns the matrix so the methods can be chained.
	     *  @langversion 3.0
	     *  @playerversion Flash 10.2
	     *  @playerversion AIR 2.6
	     *  @productversion FlexJS 0.7
		 */
		public function invert():Matrix
		{
			// if b and c are both 0, we can simplify this.
			if (b == 0 && c == 0)
			{
				a = 1 / a;
				d = 1 / d;
				tx *= -a;
				ty *= -d;
			}
			else
			{
				var det:Number = determinant();
				if (det == 0)
				{
					identity();
					return this;
				}
				det = 1 / det;
				var newa:Number = d * det;
				d = a * det;
				a = newa;
				b *= -det;
				c *= -det;
			
				var newtx:Number = - a * tx - c * ty;
				ty = - b * tx - d * ty;
				tx = newtx;
			}
			return this;
		}
		
		/**
		 *  Resets the matrix to the default values.
		 *  Returns the matrix so the methods can be chained.
	     *  @langversion 3.0
	     *  @playerversion Flash 10.2
	     *  @playerversion AIR 2.6
	     *  @productversion FlexJS 0.7
		 */
		public function identity():Matrix
		{
			a = d = 1;
			b = c = tx = ty = 0;
			return this;
		}
		
		/**
		 *  Rotates the Matrix by the specified value.
		 *  Returns the matrix so the methods can be chained.
	     *  @langversion 3.0
	     *  @playerversion Flash 10.2
	     *  @playerversion AIR 2.6
	     *  @productversion FlexJS 0.7
		 */
		public function rotate(angle:Number):Matrix
		{
			var cos:Number = Math.cos(angle);
			var sin:Number = Math.sin(angle);
			
			var newa:Number = a * cos - b * sin;
			b = a * sin + b * cos;
			a = newa;
		
			var newc:Number = c * cos - d * sin;
			d = c * sin + d * cos;
			c = newc;
		
			var newtx:Number = tx * cos - ty * sin;
			ty = tx * sin + ty * cos;
			tx = newtx;
			return this;
		}
		
		/**
		 *  Moves the Matrix by the specified amount
		 *  Returns the matrix so the methods can be chained.
	     *  @langversion 3.0
	     *  @playerversion Flash 10.2
	     *  @playerversion AIR 2.6
	     *  @productversion FlexJS 0.7
		 */
		public function translate(x:Number, y:Number):Matrix
		{
			tx += x;
			ty += y;
			return this;
		}
		
		/**
		 *  Scales the Matrix by the specified amount.
		 *  Returns the matrix so the methods can be chained.
	     *  @langversion 3.0
	     *  @playerversion Flash 10.2
	     *  @playerversion AIR 2.6
	     *  @productversion FlexJS 0.7
		 */
		public function scale(x:Number, y:Number):Matrix
		{
			a *= x;
			b *= y;
			c *= x;
			d *= y;
			tx *= x;
			ty *= y;
			return this;
		}
		
		/**
		 *  Uses the Matrix to transform the point without the translation values.
		 *  Returns a new Point. The original Point is unchanged.
	     *  @langversion 3.0
	     *  @playerversion Flash 10.2
	     *  @playerversion AIR 2.6
	     *  @productversion FlexJS 0.7
		 */
		public function deltaTransformPoint(point:Point):Point
		{
			return new Point(a * point.x + c * point.y, d * point.y + b * point.x);
		}
		
		/**
		 *  Uses the Matrix to transform the point including the translation values.
		 *  Returns a new Point. The original Point is unchanged.
	     *  @langversion 3.0
	     *  @playerversion Flash 10.2
	     *  @playerversion AIR 2.6
	     *  @productversion FlexJS 0.7
		 */
		public function transformPoint(point:Point):Point
		{
			return new Point(a * point.x + c * point.y + tx, d * point.y + b * point.x + ty);
		}
		
		/**
		 *  Returns a string representation of the Matrix.
	     *  @langversion 3.0
	     *  @playerversion Flash 10.2
	     *  @playerversion AIR 2.6
	     *  @productversion FlexJS 0.7
		 */
		public function toString():String
		{
			return "(a=" + a + ", b=" + b + ", c=" + c + ", d=" + d + ", tx=" + tx + ", ty=" + ty + ")";
		}
		
		/**
		 *  Copies the values from another Matrix.
		 *  Returns the matrix so the methods can be chained.
	     *  @langversion 3.0
	     *  @playerversion Flash 10.2
	     *  @playerversion AIR 2.6
	     *  @productversion FlexJS 0.7
		 */
		public function copyFrom(source:Matrix):Matrix
		{
			a = source.a;
			b = source.b;
			c = source.c;
			d = source.d;
			tx = source.tx;
			ty = source.ty;
			return this;
		}
		
	}
}