////////////////////////////////////////////////////////////////////////////////
//
//  Licensed to the Apache Software Foundation (ASF) under one or more
//  contributor license agreements.  See the NOTICE file distributed with
//  this work for additional information regarding copyright ownership.
//  The ASF licenses this file to You under the Apache License, Version 2.0
//  (the "License"); you may not use this file except in compliance with
//  the License.  You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
////////////////////////////////////////////////////////////////////////////////

package mx.accessibility
{

import flash.accessibility.Accessibility;
import flash.events.Event;
import mx.accessibility.AccConst;
import mx.controls.MenuBar;
import mx.core.UIComponent;
import mx.core.mx_internal;
import mx.events.MenuEvent;

use namespace mx_internal;

/**
 *  MenuBarAccImpl is a subclass of AccessibilityImplementation
 *  which implements accessibility for the MenuBar class.
 *  
 *  @langversion 3.0
 *  @playerversion Flash 9
 *  @playerversion AIR 1.1
 *  @productversion Flex 3
 */
public class MenuBarAccImpl extends AccImpl
{
    include "../core/Version.as";

	//--------------------------------------------------------------------------
	//
	//  Class methods
	//
	//--------------------------------------------------------------------------

	/**
	 *  Enables accessibility in the MenuBar class.
	 * 
	 *  <p>This method is called by application startup code
	 *  that is autogenerated by the MXML compiler.
	 *  Afterwards, when instances of MenuBar are initialized,
	 *  their <code>accessibilityImplementation</code> property
	 *  will be set to an instance of this class.</p>
	 *  
	 *  @langversion 3.0
	 *  @playerversion Flash 9
	 *  @playerversion AIR 1.1
	 *  @productversion Flex 3
	 */
	public static function enableAccessibility():void
	{
		MenuBar.createAccessibilityImplementation =
			createAccessibilityImplementation;
	}

	/**
	 *  @private
	 *  Creates a MenuBar's AccessibilityImplementation object.
	 *  This method is called from UIComponent's
	 *  initializeAccessibility() method.
	 */
	mx_internal static function createAccessibilityImplementation(
								component:UIComponent):void
	{
		component.accessibilityImplementation =
			new MenuBarAccImpl(component);
	}

	//--------------------------------------------------------------------------
	//
	//  Constructor
	//
	//--------------------------------------------------------------------------

	/**
	 *  Constructor.
	 *
	 *  @param master The UIComponent instance that this AccImpl instance
	 *  is making accessible.
	 *  
	 *  @langversion 3.0
	 *  @playerversion Flash 9
	 *  @playerversion AIR 1.1
	 *  @productversion Flex 3
	 */
	public function MenuBarAccImpl(master:UIComponent)
	{
		super(master);

		role = AccConst.ROLE_SYSTEM_MENUBAR;
	}

	//--------------------------------------------------------------------------
	//
	//  Overridden properties: AccImpl
	//
	//--------------------------------------------------------------------------

	//----------------------------------
	//  eventsToHandle
	//----------------------------------

	/**
	 *  @private
	 *	Array of events that we should listen for from the master component.
	 */
	override protected function get eventsToHandle():Array
	{
		return super.eventsToHandle.concat(
			[ "menuShow", "menuHide", "focusIn", "focusOut" ]);
	}
	
	//--------------------------------------------------------------------------
	//
	//  Overridden methods: AccessibilityImplementation
	//
	//--------------------------------------------------------------------------

	/**
	 *  Gets the role for the component.
	 *
	 *  @param childID uint
	 *  
	 *  @langversion 3.0
	 *  @playerversion Flash 9
	 *  @playerversion AIR 1.1
	 *  @productversion Flex 3
	 */
	override public function get_accRole(childID:uint):uint
	{
		return childID == 0 ? role : AccConst.ROLE_SYSTEM_MENUITEM;
	}

	/**
	 *  @private
	 *  IAccessible method for returning the state of the MenuItem.
	 *  States are predefined for all the components in MSAA.
	 *  Values are assigned to each state.
	 *  Depending upon the listItem being Selected, Selectable,
	 *  Invisible, Offscreen, a value is returned.
	 *
	 *  @param childID uint
	 *
	 *  @return State uint
	 */
	override public function get_accState(childID:uint):uint
	{
		var accState:uint = getState(childID);
		
		if (childID > 0)
		{
			var menuBar:MenuBar = MenuBar(master);

			var index:int = childID - 1;

			if (!menuBar.menuBarItems[index] || !menuBar.menuBarItems[index].enabled)
			{
				accState |= AccConst.STATE_SYSTEM_UNAVAILABLE;
			}
			else
			{
				accState |= AccConst.STATE_SYSTEM_SELECTABLE | AccConst.STATE_SYSTEM_FOCUSABLE;
				
				// if (menuBar.getMenuAt(index))
				accState |= AccConst.STATE_SYSTEM_HASPOPUP;

				if (index == menuBar.selectedIndex)
					accState |= AccConst.STATE_SYSTEM_HOTTRACKED | AccConst.STATE_SYSTEM_FOCUSED;
			}
		}
		return accState;
	}

	/**
	 *  @private
	 *  IAccessible method for returning the Default Action.
	 *
	 *  @param childID uint
	 *
	 *  @return name of default action.
	 */
	override public function get_accDefaultAction(childID:uint):String
	{
		if (childID == 0)
			return null;
		
		return childID - 1 == MenuBar(master).selectedIndex ? "Close" : "Open";
	}

	/**
	 *  @private
	 *  IAccessible method for executing the Default Action.
	 *
	 *  @param childID uint
	 */
	override public function accDoDefaultAction(childID:uint):void
	{
		if (childID > 0)
		{
			var index:int = childID - 1;
			//MenuBar(master).selectedIndex = index;
			//MenuBar(master).showMenu(index);
		}
	}

 	/**
	 *  @private
	 *  Method to return an array of childIDs.
	 *
	 *  @return Array
	 */
	override public function getChildIDArray():Array
	{
		var n:int = MenuBar(master).menuBarItems ?
					MenuBar(master).menuBarItems.length :
					0;

		return createChildIDArray(n);
	}

 	/**
	 *  @private
	 *  IAccessible method for returning the bounding box of the MenuBarItem.
	 *
	 *  @param childID uint
	 *
	 *  @return Location Object
	 */
	override public function accLocation(childID:uint):*
	{
		//should check that this is returning the needed component
		return MenuBar(master).menuBarItems[childID - 1];
		//return MenuBar(master).getMenuBarItemAt(childID - 1);
	}

	/**
	 *  @private
	 *  IAccessible method for returning the childFocus of the List.
	 *
	 *  @param childID uint
	 *
	 *  @return focused childID.
	 */
	override public function get_accFocus():uint
	{
		var index:int = MenuBar(master).selectedIndex;
		
		return index >= 0 ? index + 1 : 0;
	}

	//--------------------------------------------------------------------------
	//
	//  Overridden methods: AccImpl
	//
	//--------------------------------------------------------------------------

	/**
	 *  @private
	 *  IAccessible method for returning the name of the MenuBar
	 *  which is spoken out by the screen reader.
	 *  The MenuItem should return the label as the name
	 *  and MenuBar should return the name specified in the Accessibility Panel.
	 *
	 *  @param childID uint
	 *
	 *  @return Name String
	 */
	override protected function getName(childID:uint):String
	{
		if (childID == 0)
			return "";

		var menuBar:MenuBar = MenuBar(master);

		var index:int = childID - 1;
		
		if (menuBar.menuBarItems && menuBar.menuBarItems.length > index)
		{
			if (menuBar.menuBarItems[index] && menuBar.menuBarItems[index].data)
				return menuBar.itemToLabel(menuBar.menuBarItems[index].data);
		}

		return "";
	}

	//--------------------------------------------------------------------------
	//
	//  Overridden event handlers: AccImpl
	//
	//--------------------------------------------------------------------------

	/**
	 *  @private
	 *  Override the generic event handler.
	 *  All AccImpl must implement this
	 *  to listen for events from its master component. 
	 */
	override protected function eventHandler(event:Event):void
	{
		// Let AccImpl class handle the events
		// that all accessible UIComponents understand.
		$eventHandler(event);

		switch (event.type)
		{
			case "menuShow":
			{
				var index:int = MenuBar(master).selectedIndex;
				
				// since all the menu events are also received by Menubar.
				if (index >= 0 && !MenuEvent(event).menu.parentMenu)
				{
					var childID:uint = index + 1;

					Accessibility.sendEvent(master, childID,
											AccConst.EVENT_OBJECT_FOCUS);

					Accessibility.sendEvent(master, childID,
											AccConst.EVENT_OBJECT_SELECTION);
				}

				break;
			}

			case "menuHide":
			{
				if (!MenuEvent(event).menu.parentMenu)
					Accessibility.sendEvent(master, 0, AccConst.EVENT_SYSTEM_MENUEND);
				break;
			}

			case "focusIn":
			{
				Accessibility.sendEvent(master, 0, AccConst.EVENT_SYSTEM_MENUSTART);
				break;
			}
			
			case "focusOut":
			{
				if (MenuBar(master).selectedIndex == -1)
					Accessibility.sendEvent(master, 0, AccConst.EVENT_SYSTEM_MENUEND);
				break;
			}
		}
	}
}

}
